/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/case_conv.hpp>
#include <stdexcept>

#include "LineBased.hh"
#include "ScopedHandler.hh"
#include "SceneObject.hh"
#include "PropKeys.hh"
#include "props/FloatPropBase.hh"
#include "props/Uint32PropBase.hh"
#include "props/EnumPropBase.hh"


using namespace peekabot;


LineBased::LineBased() throw()
    : m_line_width(1),
      m_line_style(LINE_STYLE_SOLID),
      m_stipple_factor(1)
{
}


LineBased::LineBased(ScopedHandler *handler)
    : m_line_width(1),
      m_line_style(LINE_STYLE_SOLID),
      m_stipple_factor(1)
{
    // Register start tag handlers

    handler->add_start_handler(
        "line_width",
        boost::bind(
            &LineBased::line_width_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "line_style",
        boost::bind(
            &LineBased::line_style_start_handler, this, _1, _2, _3));
}


float LineBased::get_line_width() const
{
    return m_line_width;
}


void LineBased::set_line_width(float line_width)
{
    m_line_width = line_width;
    m_line_width_set_signal();
}


void LineBased::set_line_style(LineStyle style)
{
    m_line_style = style;
    m_line_style_set_signal();
}


LineStyle LineBased::get_line_style() const
{
    return m_line_style;
}


void LineBased::set_stipple_factor(boost::uint32_t factor)
{
    if( factor < 1 || factor > 256 )
        throw std::range_error(
            "Stipple factor must be in the range [1,256]");

    m_stipple_factor = factor;
    m_stipple_factor_set_signal();
}


boost::uint32_t LineBased::get_stipple_factor() const
{
    return m_stipple_factor;
}


PropMap &LineBased::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
    }

    return *s_prop_adapters;
}


void LineBased::create_prop_adapters(PropMap &adapters)
{
    class LineWidthAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            p->set_line_width(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const LineBased *p = dynamic_cast<const LineBased *>(obj);
            assert( p );
            return Any(p->get_line_width());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            return p->line_width_set_signal();
        }

        virtual float get_min_value() const { return 0.0; }
    };

    class StippleFactorAdapter : public Uint32PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            p->set_stipple_factor(any_cast<boost::uint32_t>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const LineBased *p = dynamic_cast<const LineBased *>(obj);
            assert( p );
            return Any(p->get_stipple_factor());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            return p->stipple_factor_set_signal();
        }

        virtual boost::uint32_t get_min_value() const { return 1; }

        virtual boost::uint32_t get_max_value() const { return 256; }
    };

    class LineStyleAdapter : public EnumPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            p->set_line_style(any_cast<LineStyle>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const LineBased *p = dynamic_cast<const LineBased *>(obj);
            assert( p );
            return Any(p->get_line_style());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );
            return p->line_style_set_signal();
        }

        virtual void get_enum_values(std::vector<std::string> &v) const
        {
            v.push_back("Solid");
            v.push_back("Dashed");
            v.push_back("Dotted");
            v.push_back("Dash-dot");
            v.push_back("Dash-dot-dot");
        }

        virtual void set_from_text(const std::string &text, SceneObject *obj)
        {
            LineBased *p = dynamic_cast<LineBased *>(obj);
            assert( p );

            if( text == "Solid" )
                p->set_line_style(LINE_STYLE_SOLID);
            else if( text == "Dashed" )
                p->set_line_style(LINE_STYLE_DASHED);
            else if( text == "Dotted" )
                p->set_line_style(LINE_STYLE_DOTTED);
            else if( text == "Dash-dot" )
                p->set_line_style(LINE_STYLE_DASH_DOT);
            else if( text == "Dash-dot-dot" )
                p->set_line_style(LINE_STYLE_DASH_DOT_DOT);
            else
                assert( false );
        }

        virtual std::string get_as_text(const SceneObject *obj) const
        {
            const LineBased *p = dynamic_cast<const LineBased *>(obj);
            assert( p );

            if( p->get_line_style() == LINE_STYLE_SOLID )
                return "Solid";
            else if( p->get_line_style() == LINE_STYLE_DASHED )
                return "Dashed";
            else if( p->get_line_style() == LINE_STYLE_DOTTED )
                return "Dotted";
            else if( p->get_line_style() == LINE_STYLE_DASH_DOT )
                return "Dash-dot";
            else if( p->get_line_style() == LINE_STYLE_DASH_DOT_DOT )
                return "Dash-dot-dot";
            else
                assert( false );
        }
    };

    adapters.insert(
        PropMap::value_type(
            LINE_WIDTH_PROP, new LineWidthAdapter));

    adapters.insert(
        PropMap::value_type(
            LINE_STIPPLE_FACTOR_PROP, new StippleFactorAdapter));

    adapters.insert(
        PropMap::value_type(
            LINE_STYLE_PROP, new LineStyleAdapter));
}


//
// --------------- XML handler methods --------------------
//


void LineBased::line_width_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &LineBased::line_width_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}

void LineBased::line_width_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler)
{
    set_line_width(boost::lexical_cast<float>(cdata));
}

void LineBased::line_style_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    ScopedHandler::TagScope scope;

    int factor = boost::lexical_cast<int>(attributes["factor"]);

    scope.cdata_functor = boost::bind(
        &LineBased::line_style_cdata_handler, this, _1, _2, factor);

    handler->enter_scope(scope);
}

void LineBased::line_style_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler,
    int factor)
{
    std::string style = cdata;
    boost::to_lower(style);

    set_stipple_factor(factor);

    if( style == "solid" )
        set_line_style(LINE_STYLE_SOLID);
    else if( style == "dashed" )
        set_line_style(LINE_STYLE_DASHED);
    else if( style == "dotted" )
        set_line_style(LINE_STYLE_DOTTED);
    else if( style == "dash-dot" )
        set_line_style(LINE_STYLE_DASH_DOT);
    else if( style == "dash-dot-dot" )
        set_line_style(LINE_STYLE_DASH_DOT_DOT);
    else
        throw std::runtime_error("Unknown line style");
}
