/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_LINE_BASED_HH_INCLUDED
#define PEEKABOT_LINE_BASED_HH_INCLUDED


#include <boost/signals2.hpp>
#include <boost/cstdint.hpp>

#include "Types.hh"
#include "XMLHandler.hh"
#include "PropMap.hh"


namespace peekabot
{

    class ScopedHandler;

    /**
     * \internal
     *
     * \brief An abstraction of objects in the scene that are scalable.
     *
     * A non-instantiable class that adds to scene objects the scale
     * property and methods for manipulating it. Classes that need
     * this property inherit it from this class.
     */
    class LineBased
    {
    public:
        LineBased() throw();

        /**
         * \brief Construct a line-based object using an XML document handler.
         */
        LineBased(ScopedHandler *handler);

        virtual ~LineBased() {}

        float get_line_width() const;

        void set_line_width(float line_width);

        void set_line_style(LineStyle style);

        LineStyle get_line_style() const;

        void set_stipple_factor(boost::uint32_t factor);

        boost::uint32_t get_stipple_factor() const;

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> LineWidthSetSignal;

        typedef boost::signals2::signal<void ()> LineStyleSetSignal;

        typedef boost::signals2::signal<void ()> StippleFactorSetSignal;

        inline LineWidthSetSignal &line_width_set_signal() const
        {
            return m_line_width_set_signal;
        }

        inline LineStyleSetSignal &line_style_set_signal() const
        {
            return m_line_style_set_signal;
        }

        inline StippleFactorSetSignal &stipple_factor_set_signal() const
        {
            return m_stipple_factor_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /**
         * \brief <line_width> start handler for XML parsing.
         */
        void line_width_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /**
         *\brief <line_width> cdata handler for XML parsing.
         */
        void line_width_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler);

        /**
         * \brief <line_style> start handler for XML parsing.
         */
        void line_style_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler);

        /**
         *\brief <line_style> cdata handler for XML parsing.
         */
        void line_style_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler,
            int factor);

        /// @}

    private:
        float m_line_width;

        LineStyle m_line_style;

        int m_stipple_factor;

        mutable LineWidthSetSignal m_line_width_set_signal;
        mutable LineStyleSetSignal m_line_style_set_signal;
        mutable StippleFactorSetSignal m_stipple_factor_set_signal;
    };
}


#endif // PEEKABOT_LINE_BASED_HH_INCLUDED
