/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_MESSAGE_HUB_HH
#define __PEEKABOT_MESSAGE_HUB_HH


#include <map>
#include <set>
#include <string>
#include <boost/thread/recursive_mutex.hpp>

#include "Types.hh"
#include "Singleton.hh"

namespace peekabot
{

    /**
     * \internal
     *
     * \brief Enumerates all the different categories that a message 
     * can be of.
     */
    enum MessageType
    {
        /**
         * \brief Messages that somehow originate from a clients belong to 
         * this category.
         */
        CLIENT_MESSAGE = 0,

        /**
         * \brief Messages of purely informational nature.
         *
         * An example could be "GL_ARB_vertex_buffer_object not present, falling 
         * back on vertex arrays".
         */
        INFO_MESSAGE = 1,

        /**
         * \brief Warning messages.
         *
         * A notice about an action that failed to execute is a typical example.
         * They shouldn't contain information that is critical to the typical 
         * end user and will generally not be read unless actively searching
         * for incorrect or unexpected behaviour.
         */
        WARNING_MESSAGE = 2,

        /**
         * \brief Error messages.
         */
        ERROR_MESSAGE = 3,

        /**
         * \brief Debug messages. Use for debugging print outs only.
         */
        DEBUG_MESSAGE = 4,

        /**
         * \brief Messages of purely informational nature, but not intended for
         * display to the user.
         *
         * An example could be logging the available OpenGL extensions.
         */
        LOG_MESSAGE = 5
    };


    class Subscriber;
    class Message;


    /**
     * \internal
     *
     * \brief A facility for subscribing to and publishing messages of various
     * types - debug and client messages, for example.
     *
     * The good that comes out of introducing this class is decoupling of the 
     * message sender and source - for example, the rendering component no 
     * longer has to have knowledge of any GUI interfaces to write messages to 
     * the console. 
     */
    class MessageHub
    {
        /**
         * \brief Typedef for the datatype used to store the set of subscribers
         * associated with a \c MessageType.
         */
        typedef std::set<Subscriber *> SubscriberSet;

        /**
         * \brief Typedef for the datatype used to store the mapping from 
         * message type to a set of subscribers.
         */
        typedef std::map<MessageType, SubscriberSet> SubscriberMap;

        /**
         * \brief Stores all subscribers as a mapping from message types to
         * sets of subscribers.
         *
         * \remarks Access is governed by \c m_mutex, on which a lock must be
         * held for access.
         */
        SubscriberMap m_subscribers;

        /**
         * \brief Used to govern read and write access to all of the classes 
         * members.
         */
        mutable boost::recursive_mutex m_mutex;

    public:
        ~MessageHub();

        /**
         * \brief Publish a message.
         *
         * Publish a message that will be sent to all parties subscribing to
         * messages of the type published. The call will block until all 
         * recipients has been notified and handled the message. Note that this
         * implies that a class can both be in the midst of publishing and 
         * handling a message at the same time, something you have to consider
         * if you have a class that is both a message source and recipient.
         *
         * \pre The formatted message must be shorter than 2048 characters.
         *
         * \param type The type of the published message.
         * \param msg The message format string, in the same format as 
         *            printf format strings. E.g. "foo: %d" followed by the 
         *            additional argument 10 will yield "foo: 10".
         */
        void publish(
            MessageType type, 
            const std::string &msg, ...) const throw();

        /**
         * \brief Publish a message, with information regarding the source of 
         * the message.
         *
         * \param type The type of the published message.
         * \param source The source of the message.
         * \param msg The message format string, in the same format as 
         *            printf format strings. E.g. "foo: %d" followed by the 
         *            additional argument 10 will yield "foo: 10" being 
         *            published.
         *
         * \sa <tt>publish(MessageType, const std::string &)</tt>
         */
        void publish(
            MessageType type, SourceID source, 
            const std::string &msg, ...) const throw();

        /**
         * \brief Subscribe to messages of the given category.
         *
         * If the specified recipient already has an active subscription on 
         * messages of the given type calling this method will not do anything.
         *
         * \param type The type of messages the \c recipient wishes to recieve.
         * \param recipient The recipient of the subscription.
         */
        void subscribe(MessageType type, Subscriber *recipient) throw();

        /**
         * \brief Subscribe to all messages.
         *
         * \param recipient The subscriber whom to send all messages to.
         */
        void subscribe(Subscriber *recipient) throw();

        /**
         * \brief Unsubscribe to a message category.
         *
         * If the given subscriber isn't subscribing to the specified message 
         * category this method will do nothing.
         *
         * \param type The type of messages that's no longer wanted.
         * \param recipient The affected subscriber.
         */
        void unsubscribe(MessageType type, Subscriber *recipient) throw();

    protected:
        /**
         * \brief Send the given message to all affected subscribers.
         *
         * \param message The message to be sent.
         */
        void publish(const Message &message) const throw();
    };



    /**
     * \brief A singleton for accessing a the message hub used by all, as
     * there should be only a single, shared message hub.
     */
    typedef Singleton<MessageHub> TheMessageHub;


}


#endif // __PEEKABOT_MESSAGE_HUB_HH
