/*
 * Copyright Anders Boberg 2007.
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "ModelObject.hh"
#include "Path.hh"
#include "ObjectTypes.hh"

#include <boost/bind.hpp>
#include <boost/filesystem/path.hpp>


using namespace peekabot;


HandlerInformer ModelObject::ms_handler_informer(
    "model", &ModelObject::start_handler);


ModelObject::ModelObject(const Path &model_path)
    : SceneObject("model")
{
    set_model_path(model_path);
    set_color(RGBColor(1,1,1));
}


ModelObject::ModelObject(ScopedHandler *handler)
    : SceneObject("model", handler),
      ScalableObject(handler)
{
    set_color(RGBColor(1,1,1));

    // Register the file tag start handler
    handler->add_start_handler(
        "file",
        boost::bind(&ModelObject::file_start_handler,
                    this, _1, _2, _3));
}


const Path &ModelObject::get_model_path() const
{
    return m_model_path;
}


void ModelObject::set_model_path(const Path &path)
{
    m_model_path = path;
}


void ModelObject::accept(ObjectVisitor* visitor) throw()
{
    visitor->visit(this);
}


ObjectType ModelObject::get_object_type() const
{
    return MODEL_OBJECT;
}


PropMap &ModelObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, ScalableObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void ModelObject::create_prop_adapters(PropMap &adapters)
{
}


//
// --------------- XML handler methods --------------------
//


void ModelObject::start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new ModelObject(handler);

    // Set the new object as the current object
    ScopedMap & variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}

void ModelObject::file_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope file_scope;

    file_scope.cdata_functor = boost::bind(
        &ModelObject::file_cdata_handler, this, _1, _2);

    handler->enter_scope(file_scope);
}

void ModelObject::file_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler)
{
    ScopedMap &variables = handler->get_variables();

    assert( variables.exists<boost::filesystem::path>("current_file_dir") );

    std::list<boost::filesystem::path> model_dirs;
    model_dirs.push_back(
        variables.get_variable<boost::filesystem::path>("current_file_dir"));

    assert( variables.exists<Path>("scene_file_path") );
    Path p = variables.get_variable<Path>("scene_file_path");
    p.set_path(cdata);
    set_model_path(p);
}
