/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "OccupancyGrid2D.hh"
#include "PropKeys.hh"
#include "props/ColorPropBase.hh"
#include "props/FloatPropBase.hh"
#include "ObjectTypes.hh"

#include <boost/bind.hpp>


using namespace peekabot;


/*HandlerInformer OccupancyGrid2D::ms_handler_informer(
    "occupancy_grid_2d", &OccupancyGrid2D::start_handler);*/


OccupancyGrid2D::OccupancyGrid2D(float cell_size) throw()
    : SceneObject("occupancy_grid_2d"),
      m_cell_size(cell_size),
      m_unoccupied_color(0.9, 0.9, 0.9),
      m_occupied_color(0, 0, 0)
{
}


OccupancyGrid2D::OccupancyGrid2D(ScopedHandler *handler)
    : SceneObject("occupancy_grid_2d", handler)
{
    // Register the file tag start handler
    /*handler->add_start_handler(
        "file",
        boost::bind(&OccupancyGrid2D::file_start_handler,
        this, _1, _2, _3));*/
}


void OccupancyGrid2D::accept(ObjectVisitor *visitor) throw()
{
    visitor->visit(this);
}


ObjectType OccupancyGrid2D::get_object_type() const
{
    return OG2D_OBJECT;
}


float OccupancyGrid2D::get_cell_size() const
{
    return m_cell_size;
}


const RGBColor &OccupancyGrid2D::get_unoccupied_color() const
{
    return m_unoccupied_color;
}


void OccupancyGrid2D::set_unoccupied_color(const RGBColor &color)
{
    m_unoccupied_color = color;
    m_unoccupied_color_set_signal();
}


const RGBColor &OccupancyGrid2D::get_occupied_color() const
{
    return m_occupied_color;
}


void OccupancyGrid2D::set_occupied_color(const RGBColor &color)
{
    m_occupied_color = color;
    m_occupied_color_set_signal();
}


void OccupancyGrid2D::set_cell(const Eigen::Vector2f &x, float belief) throw()
{
    std::vector<std::pair<Eigen::Vector2f, float> > cells;
    cells.push_back(std::make_pair(x, belief));
    set_cells(cells);
}


void OccupancyGrid2D::set_cells(
    const std::vector<std::pair<Eigen::Vector2f, float> > &cells) throw()
{
    m_cells_set_signal(cells);
}


PropMap &OccupancyGrid2D::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void OccupancyGrid2D::create_prop_adapters(PropMap &adapters)
{
    class UnoccupiedColorAdapter : public ColorPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            OccupancyGrid2D *p = dynamic_cast<OccupancyGrid2D *>(obj);
            assert( p );
            p->set_unoccupied_color(any_cast<RGBColor>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid2D *p =
                dynamic_cast<const OccupancyGrid2D *>(obj);
            assert( p );
            return Any(p->get_unoccupied_color());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            OccupancyGrid2D *p = dynamic_cast<OccupancyGrid2D *>(obj);
            assert( p );
            return p->unoccupied_color_set_signal();
        }
    };

    class OccupiedColorAdapter : public ColorPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            OccupancyGrid2D *p = dynamic_cast<OccupancyGrid2D *>(obj);
            assert( p );
            p->set_occupied_color(any_cast<RGBColor>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid2D *p =
                dynamic_cast<const OccupancyGrid2D *>(obj);
            assert( p );
            return Any(p->get_occupied_color());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            OccupancyGrid2D *p = dynamic_cast<OccupancyGrid2D *>(obj);
            assert( p );
            return p->occupied_color_set_signal();
        }
    };


    class CellSizeAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) {}

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid2D *p =
                dynamic_cast<const OccupancyGrid2D *>(obj);
            assert( p );
            return Any(p->get_cell_size());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            static SignalType s_dummy_signal;
            return s_dummy_signal;
        }

        virtual bool is_read_only() const { return true; }
    };


    adapters.insert(
        PropMap::value_type(OG2D_UNOCCUPIED_COLOR_PROP,
                            new UnoccupiedColorAdapter));

    adapters.insert(
        PropMap::value_type(OG2D_OCCUPIED_COLOR_PROP,
                            new OccupiedColorAdapter));

    adapters.insert(
        PropMap::value_type(OG2D_CELL_SIZE_PROP,
                            new CellSizeAdapter));
}

//
// --------------- XML handler methods --------------------
//

void OccupancyGrid2D::start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new OccupancyGrid2D(handler);

    // Set the new object as the current object
    ScopedMap & variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}
