/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "OccupancyGrid3D.hh"
#include "ObjectTypes.hh"
#include "PropKeys.hh"
#include "props/BoolPropBase.hh"
#include "props/FloatPropBase.hh"

#include <boost/bind.hpp>


using namespace peekabot;


/*HandlerInformer OccupancyGrid3D::ms_handler_informer(
    "occupancy_grid_2d", &OccupancyGrid3D::start_handler);*/


OccupancyGrid3D::OccupancyGrid3D(float cell_xy_size, float cell_z_size) throw()
    : SceneObject("occupancy_grid_3d"),
      m_cell_xy_size(cell_xy_size),
      m_cell_z_size(cell_z_size),
      m_color_mapping_enabled(false),
      m_color_mapping_z_min(0),
      m_color_mapping_z_max(0)
{
}


OccupancyGrid3D::OccupancyGrid3D(ScopedHandler *handler)
    : SceneObject("occupancy_grid_3d", handler),
      m_color_mapping_enabled(false),
      m_color_mapping_z_min(0),
      m_color_mapping_z_max(0)
{
    // Register the file tag start handler
    /*handler->add_start_handler(
        "file",
        boost::bind(&OccupancyGrid3D::file_start_handler,
        this, _1, _2, _3));*/
}


void OccupancyGrid3D::accept(ObjectVisitor *visitor) throw()
{
    visitor->visit(this);
}


ObjectType OccupancyGrid3D::get_object_type() const
{
    return OG3D_OBJECT;
}


float OccupancyGrid3D::get_cell_xy_size() const throw()
{
    return m_cell_xy_size;
}


float OccupancyGrid3D::get_cell_z_size() const throw()
{
    return m_cell_z_size;
}


void OccupancyGrid3D::set_cell(const Eigen::Vector3f &x, float belief) throw()
{
    std::vector<std::pair<Eigen::Vector3f, float> > cells;
    cells.push_back(std::make_pair(x, belief));
    set_cells(cells);
}


void OccupancyGrid3D::set_cells(
    const std::vector<std::pair<Eigen::Vector3f, float> > &cells) throw()
{
    m_cells_set_signal(cells);
}


void OccupancyGrid3D::toggle_color_mapping(bool enable)
{
    m_color_mapping_enabled = enable;
    m_color_mapping_set_signal();
}


void OccupancyGrid3D::start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new OccupancyGrid3D(handler);

    // Set the new object as the current object
    ScopedMap & variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


bool OccupancyGrid3D::is_color_mapping_enabled() const
{
    return m_color_mapping_enabled;
}


void OccupancyGrid3D::set_color_mapping_z_min(float z_min)
{
    m_color_mapping_z_min = z_min;
    m_color_mapping_z_min_set_signal();
}


void OccupancyGrid3D::set_color_mapping_z_max(float z_max)
{
    m_color_mapping_z_max = z_max;
    m_color_mapping_z_max_set_signal();
}


float OccupancyGrid3D::get_color_mapping_z_min() const
{
    return m_color_mapping_z_min;
}


float OccupancyGrid3D::get_color_mapping_z_max() const
{
    return m_color_mapping_z_max;
}


PropMap &OccupancyGrid3D::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void OccupancyGrid3D::create_prop_adapters(PropMap &adapters)
{
    class CellXYSizeAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) { assert(false); }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return Any(p->get_cell_xy_size());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            static SignalType s_dummy_signal;
            return s_dummy_signal;
        }

        virtual bool is_read_only() const { return true; }
    };

    class CellZSizeAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) { assert(false); }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return Any(p->get_cell_z_size());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            static SignalType s_dummy_signal;
            return s_dummy_signal;
        }

        virtual bool is_read_only() const { return true; }
    };

    class ColorMappingEnabledAdapter : public BoolPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            OccupancyGrid3D *p = dynamic_cast<OccupancyGrid3D *>(obj);
            assert( p );
            p->toggle_color_mapping(any_cast<bool>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return Any(p->is_color_mapping_enabled());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return p->color_mapping_set_signal();
        }
    };

    class ColorMappingZMinAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            OccupancyGrid3D *p = dynamic_cast<OccupancyGrid3D *>(obj);
            assert( p );
            p->set_color_mapping_z_min(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return Any(p->get_color_mapping_z_min());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return p->color_mapping_z_min_set_signal();
        }
    };

    class ColorMappingZMaxAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            OccupancyGrid3D *p = dynamic_cast<OccupancyGrid3D *>(obj);
            assert( p );
            p->set_color_mapping_z_max(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return Any(p->get_color_mapping_z_max());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            const OccupancyGrid3D *p =
                dynamic_cast<const OccupancyGrid3D *>(obj);
            assert( p );
            return p->color_mapping_z_max_set_signal();
        }
    };

    adapters.insert(
        PropMap::value_type(
            OG3D_CELL_XY_SIZE_PROP, new CellXYSizeAdapter));

    adapters.insert(
        PropMap::value_type(
            OG3D_CELL_Z_SIZE_PROP, new CellZSizeAdapter));

    adapters.insert(
        PropMap::value_type(
            OG3D_COLOR_MAPPING_ENABLED_PROP, new ColorMappingEnabledAdapter));

    adapters.insert(
        PropMap::value_type(
            OG3D_COLOR_MAPPING_Z_MIN_PROP, new ColorMappingZMinAdapter));

    adapters.insert(
        PropMap::value_type(
            OG3D_COLOR_MAPPING_Z_MAX_PROP, new ColorMappingZMaxAdapter));
}
