/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_OCCUPANCY_GRID_3D_HH_INCLUDED
#define PEEKABOT_OCCUPANCY_GRID_3D_HH_INCLUDED

#include <vector>
#include <Eigen/Core>

#include "SceneObject.hh"
#include "ObjectVisitor.hh"
#include "HandlerInformer.hh"


namespace peekabot
{
    class ScopedHandler;

    class OccupancyGrid3D : public SceneObject
    {
    public:
        OccupancyGrid3D(float cell_xy_size, float cell_z_size) throw();

        /** \brief Construct object from XML data.
         */
        OccupancyGrid3D(ScopedHandler *handler);

        /** \brief Method for accepting \c ObjectVisitor objects.
         *
         * This method performs an automatic type cast by calling the \c visit()
         *  method of the visitor, allowing it to perform type-dependent task.
         */
        virtual void accept(ObjectVisitor *visitor) throw();

        virtual ObjectType get_object_type() const;

        float get_cell_xy_size() const throw();

        float get_cell_z_size() const throw();

        // belief = -1 <=> clear cell
        void set_cell(const Eigen::Vector3f &x, float belief) throw();

        void set_cells(const std::vector<std::pair<Eigen::Vector3f, float> > &cells) throw();

        void toggle_color_mapping(bool enable);

        bool is_color_mapping_enabled() const;

        void set_color_mapping_z_min(float z_min);

        void set_color_mapping_z_max(float z_max);

        float get_color_mapping_z_min() const;

        float get_color_mapping_z_max() const;

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<
            void (const std::vector<std::pair<Eigen::Vector3f, float> > &cells)
            > CellsSetSignal;

        typedef boost::signals2::signal<void ()> ColorMappingSetSignal;

        typedef boost::signals2::signal<void ()> ColorMappingZMinSetSignal;

        typedef boost::signals2::signal<void ()> ColorMappingZMaxSetSignal;

        inline CellsSetSignal &cells_set_signal() const
        {
            return m_cells_set_signal;
        }

        inline ColorMappingSetSignal &color_mapping_set_signal() const
        {
            return m_color_mapping_set_signal;
        }

        inline ColorMappingZMinSetSignal &color_mapping_z_min_set_signal() const
        {
            return m_color_mapping_z_min_set_signal;
        }

        inline ColorMappingZMaxSetSignal &color_mapping_z_max_set_signal() const
        {
            return m_color_mapping_z_max_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /** \brief Element start handler for XML parsing.
         */
        static void start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /// @}

    private:
        //static HandlerInformer ms_handler_informer;

        float m_cell_xy_size;
        float m_cell_z_size;

        bool m_color_mapping_enabled;
        float m_color_mapping_z_min;
        float m_color_mapping_z_max;


        mutable CellsSetSignal m_cells_set_signal;
        mutable ColorMappingSetSignal m_color_mapping_set_signal;
        mutable ColorMappingZMinSetSignal m_color_mapping_z_min_set_signal;
        mutable ColorMappingZMaxSetSignal m_color_mapping_z_max_set_signal;
    };
}


#endif // PEEKABOT_OCCUPANCY_GRID_3D_HH_INCLUDED
