/*
 * Copyright Staffan Gimåker 2007, 2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cassert>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/format.hpp>

#include "PrintfFormatter.hh"


using namespace peekabot;



static const std::string type_to_text[] = {
    "CLIENT",
    "INFO",
    "WARNING",
    "ERROR",
    "DEBUG",
    "LOG"
};



PrintfFormatter::PrintfFormatter(const std::string &format) throw()
    : m_format(format)
{
}


std::string PrintfFormatter::format(const Message &m) const throw()
{
    std::string str;
    std::string::size_type prev_pos = 0;
    std::string::size_type pos = 0;

    while( (pos = m_format.find("%", pos)) != std::string::npos )
    {
        if( pos+1 >= m_format.size() )
            break; // End of format string reached

        if( pos > 0 && prev_pos != pos )
            str += m_format.substr(prev_pos, pos - prev_pos);

        // Extract character and advance position marker one step
        char ch = m_format[++pos];

        switch( ch )
        {
            case 'm':
                str += m.get_message();
                break;

            case 't':
                str += to_simple_string(m.get_timestamp());
                break;

            case 's':
                if( m.has_source() )
                    str += to_string(m.get_source());
                else
                    str += "???";
                break;

            case 'y':
                assert( m.get_type() >= 0 && m.get_type() <= LOG_MESSAGE );
                str += type_to_text[m.get_type()];
                break;

            default:
                str += std::string("%") + ch;
                break;
        }

        prev_pos = pos+1;
    }

    // Append the rest of format string
    if( prev_pos < m_format.size() )
        str += m_format.substr(prev_pos);

    return str;
}


std::string PrintfFormatter::to_string(const SourceID &s) const throw()
{
    std::string str;

    if( s == SERVER_SOURCE_ID )
        return "SERVER";
    else if( s == GUI_SOURCE_ID )
        return "GUI";
    else if( s >= 1000 )
        return (boost::format("CLIENT %1%") % s).str();
    else
        return "???";
}
