/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_PRINTF_FORMATTER_HH
#define __PEEKABOT_PRINTF_FORMATTER_HH


#include "MessageFormatter.hh"


namespace peekabot
{

    /**
     * \internal
     *
     * \brief A MessageFormatter providing flexible formatting options similar
     * in form and function to what's provided by C's printf function.
     */
    class PrintfFormatter : public MessageFormatter
    {
        /**
         * \brief The format string used to format the message.
         *
         * \sa PrintfFormatter(const std::string &)
         */
        const std::string m_format;

    public:
        /**
         * \brief Construct a formatter using the given format string.
         *
         * \param format A formatted message will consist of the \c format
         * string, but with certain special markers replaced with message- 
         * and/or meta-data. Special formatting commands are the following:
         * \li \%m - replaced with the message's text.
         * \li \%t - replaced with the message's timestamp (in 
         *           <tt>YYYY-mmm-DD HH:MM:SS.ffffff</tt> format, where 
         *           <tt>mmm</tt> 3 char month name. Fractional seconds only 
         *           included if non-zero.).
         * \li \%s - replaced with a textual representation of the message's 
         *           source (SERVER, CAPTURE<em>i</em>, CLIENT<em>i</em> or
         *           ??? if unknown).
         * \li \%y - replaced with a textual representation of the message's 
         *           type (CLIENT, SYSTEM, WARNING, ERROR or DEBUG).
         */
        PrintfFormatter(const std::string &format) throw();

        virtual std::string format(const Message &m) const throw();


    private:
        /**
         * \brief Get the textual representation of a source ID.
         *
         * Only client and capture sources include the ID number.
         */
        std::string to_string(const SourceID &s) const throw();
    };

}


#endif // __PEEKABOT_PRINTF_FORMATTER_HH
