/*
 * Copyright Staffan Gimåker 2006-2007, 2009-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <sstream>
#include <boost/bind.hpp>

#include "ScalableObject.hh"
#include "ScopedHandler.hh"
#include "SceneObject.hh"
#include "PropKeys.hh"
#include "props/Vector3PropBase.hh"


using namespace peekabot;


ScalableObject::ScalableObject() throw()
    : m_scale(1, 1, 1)
{
}


const Eigen::Vector3f &ScalableObject::get_scale() const
{
    return m_scale;
}


float ScalableObject::get_x_scale() const throw()
{
    return m_scale(0);
}


float ScalableObject::get_y_scale() const throw()
{
    return m_scale(1);
}


float ScalableObject::get_z_scale() const throw()
{
    return m_scale(2);
}


void ScalableObject::set_scale(const Eigen::Vector3f &scale) throw()
{
    m_scale = scale;
    m_scale_set_signal();
}


void ScalableObject::set_scale(
    float x_scale, float y_scale, float z_scale) throw()
{
    set_scale(Eigen::Vector3f(x_scale, y_scale, z_scale));
}


void ScalableObject::set_x_scale(float x_scale) throw()
{
    set_scale(x_scale, get_y_scale(), get_z_scale());
}


void ScalableObject::set_y_scale(float y_scale) throw()
{
    set_scale(get_x_scale(), y_scale, get_z_scale());
}


void ScalableObject::set_z_scale(float z_scale) throw()
{
    set_scale(get_x_scale(), get_y_scale(), z_scale);
}


PropMap &ScalableObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
    }

    return *s_prop_adapters;
}


void ScalableObject::create_prop_adapters(PropMap &adapters)
{
    class ScaleAdapter : public Vector3PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            ScalableObject *p = dynamic_cast<ScalableObject *>(obj);
            assert( p );
            p->set_scale(any_cast<Eigen::Vector3f>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const ScalableObject *p = dynamic_cast<const ScalableObject *>(obj);
            assert( p );
            return Any(p->get_scale());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            ScalableObject *p = dynamic_cast<ScalableObject *>(obj);
            assert( p );
            return p->scale_set_signal();
        }
    };

    adapters.insert(PropMap::value_type(SCALE_PROP, new ScaleAdapter));
}


//
// --------------- XML handler methods --------------------
//


ScalableObject::ScalableObject(ScopedHandler *handler)
    : m_scale(1, 1, 1)
{
    // Register the <scale> tag start handler
    handler->add_start_handler(
        "scale",
        boost::bind(&ScalableObject::scale_start_handler, this, _1, _2, _3));
}


void ScalableObject::scale_start_handler(
    const std::string & name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &ScalableObject::scale_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}


void ScalableObject::scale_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    std::stringstream s(cdata);
    float x, y, z;
    s >> x >> y >> z;
    set_scale(x, y, z);
}
