/*
 * Copyright Staffan Gimåker 2007, 2009-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_SCALABLE_OBJECT_HH_INCLUDED
#define PEEKABOT_SCALABLE_OBJECT_HH_INCLUDED


#include <Eigen/Core>
#include <boost/signals2.hpp>

#include "Types.hh"
#include "XMLHandler.hh"
#include "PropMap.hh"


namespace peekabot
{
    class ScopedHandler;

    /**
     * \internal
     *
     * \brief An abstraction of objects in the scene that are scalable.
     *
     * A non-instantiable class that adds to scene objects the scale
     * property and methods for manipulating it. Classes that need
     * this property inherit it from this class.
     */
    class ScalableObject
    {
    public:
        ScalableObject() throw();

        /** \brief Construct a scalable object using an XML document handler.
         */
        ScalableObject(ScopedHandler *handler);

        virtual ~ScalableObject() {}

        const Eigen::Vector3f &get_scale() const;

        /** \brief Returns the scaling factor along the x axis.
         */
        float get_x_scale() const throw();

        /** \brief Returns the scaling factor along the y axis.
         */
        float get_y_scale() const throw();

        /** \brief Returns the scaling factor along the z axis.
         */
        float get_z_scale() const throw();

        void set_scale(const Eigen::Vector3f &scale) throw();

        void set_scale(float x_scale, float y_scale, float z_scale) throw();

        /** \brief Set the scaling along the x axis.
         * \exception AccessDenied The object is read-only, from the context it
         *            was accesed from.
         * \param x_scale The scaling factor.
         */
        void set_x_scale(float x_scale) throw();


        /** \brief Set the scaling along the y axis.
         * \exception AccessDenied The object is read-only, from the context it
         *            was accesed from.
         * \param y_scale The scaling factor.
         */
        void set_y_scale(float y_scale) throw();

        /** \brief Set the scaling along the z axis.
         * \exception AccessDenied The object is read-only, from the context it
         *            was accesed from.
         * \param z_scale The scaling factor.
         */
        void set_z_scale(float z_scale) throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> ScaleSetSignal;

        // ---

        inline ScaleSetSignal &scale_set_signal() const
        {
            return m_scale_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /** \brief <scale> start handler for XML parsing.
         */
        void scale_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief <scale> cdata handler for XML parsing
         */
        void scale_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler) throw();

        /// @}

    private:
        Eigen::Vector3f m_scale;

        mutable ScaleSetSignal m_scale_set_signal;
    };
}


#endif // PEEKABOT_SCALABLE_OBJECT_HH_INCLUDED
