/*
 * Copyright Anders Boberg 2007.
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/cstdint.hpp>
#include <boost/format.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/lexical_cast.hpp>

#include "SceneFileLoader.hh"
#include "GroupObject.hh"
#include "HandlerInformer.hh"
#include "MessageHub.hh"
#include "ScopedMap.hh"
#include "Types.hh"
#include "SceneObject.hh"
#include "XercesParser.hh"
#include "ScopedHandler.hh"
#include "FsToolbox.hh"
#include "Config.hh"
#include "Path.hh"


using namespace peekabot;


namespace
{
    ScopedHandler::StartFunctorMap *s_global_start_handlers = 0;

    ScopedHandler::StartFunctorMap &global_start_handlers()
    {
        if( s_global_start_handlers == 0 )
            s_global_start_handlers = new ScopedHandler::StartFunctorMap;
        return *s_global_start_handlers;
    }


    void do_nothing(const std::string &, ScopedHandler *) {}
}


HandlerInformer SceneFileLoader::ms_scene_informer(
    "scene",
    &SceneFileLoader::scene_start_handler);

HandlerInformer SceneFileLoader::ms_color_informer(
    "color",
    &SceneFileLoader::color_start_handler);

HandlerInformer SceneFileLoader::ms_layer_informer(
    "layer",
    &SceneFileLoader::layer_start_handler);

HandlerInformer SceneFileLoader::ms_external_informer(
    "external",
    &SceneFileLoader::external_start_handler);

HandlerInformer SceneFileLoader::ms_include_informer(
    "include",
    &SceneFileLoader::include_start_handler);


std::set<boost::filesystem::path> SceneFileLoader::ms_included_scenes;



std::vector<SceneObject *> SceneFileLoader::load_scene(
    const Config &config, const Path &path)
{
    boost::filesystem::path exp_path = path.get_expanded_path();

    // Check for circular includes
    if( ms_included_scenes.count(exp_path) > 0 )
        throw std::runtime_error(
            "Circular include detected! Scene file '" + exp_path.string() +
            "' was previously included.");

    GroupObject foo;
    ScopedHandler handler(
        boost::bind(&SceneFileLoader::start_of_document, _1, "scene"),
        &SceneFileLoader::end_of_document);


    // Make sure parent is the root object for objects loaded
    // from the scene file.
    ScopedMap &variables = handler.get_variables();
    variables.push_variable("root_object", (SceneObject *)&foo);

    variables.push_variable("scene_file_path", path);

    // Store the current directory of scene file being loaded as a variable
    variables.push_variable(
        "current_file_dir", exp_path.branch_path());

    variables.push_variable("config", &config);

    ms_included_scenes.insert(exp_path);

    try
    {
        XercesParser parser(&handler);
        const boost::filesystem::path schema_dir =
            config.get<boost::filesystem::path>(
                "resources.share_path",
                fs::get_pkgdata_path()) / "schemas";
        parser.set_schema((schema_dir / "scene.xsd").string());
        parser.invoke(exp_path.string());

        const std::map<SceneObject *, std::string> &orig_names =
        variables.get_variable<std::map<SceneObject *, std::string> >(
            "orig_names");

        std::vector<SceneObject *> ret;
        SceneObject::Children &children = foo.get_children();
        while( !children.empty() )
        {
            SceneObject *curr = *children.begin();
            curr->detach();

            // Change the name back to the default name (if it wasn't explicitly
            // named) or the name given by <name>. We have to do this to not get
            // auto-enumeration erroneously applied twice.
            std::map<SceneObject *, std::string>::const_iterator it =
                orig_names.find(curr);

            assert( it != orig_names.end() );

            curr->set_name(it->second);
            ret.push_back(curr);
        }

        ms_included_scenes.erase(exp_path);

        return ret;
    }
    catch(...)
    {
        ms_included_scenes.erase(exp_path);
        throw;
    }
}


void SceneFileLoader::start_of_document(
    ScopedHandler* handler, const std::string &top_level_tag)
{
    ScopedHandler::TagScope scope;

    ScopedHandler::ElementStartFunctor start = get_creator(top_level_tag);

    scope.start_functors.insert(std::make_pair(top_level_tag, start));
    handler->enter_scope(scope);

    // This temporary object will be the parent of all objects declared in the
    // top level of the document. At the end tag, its children are moved into
    // the actual scene.
    ScopedMap & variables = handler->get_variables();
    SceneObject* tmp = new GroupObject();
    variables.push_variable("current_object", tmp);

    // This shortcut to the temporary root is used by the end and failure handlers
    variables.push_variable("temporary_object", tmp);
    variables.push_variable("object_counter", (unsigned int)0);

    // A map used to store original names for all added objects.
    // It's used to get correct auto-enumeration for unnamed objects.
    variables.push_variable(
        "orig_names", std::map<SceneObject *, std::string>());
}

void SceneFileLoader::end_of_document(ScopedHandler* handler)
{
    ScopedMap & variables = handler->get_variables();
    SceneObject* tmp = variables.get_variable<SceneObject*>("temporary_object");

    SceneObject *root =
        variables.get_variable<SceneObject *>("root_object");

    // Put the objects in the actual scene.
    SceneObject::Children children = tmp->get_children();
    for( SceneObject::ChildIterator it = children.begin();
         it != children.end(); ++it )
    {
        (*it)->detach();
        root->attach(*it);
    }

    unsigned int c = variables.get_variable<unsigned int>("object_counter");
    TheMessageHub::instance().publish(
        INFO_MESSAGE, "Added %d objects.", c);

    // Clean up
    delete tmp;
    variables.pop_variable<SceneObject *>("current_object");
    variables.pop_variable<SceneObject *>("temporary_object");
    variables.pop_variable<SceneObject *>("root_object");
}

void SceneFileLoader::failure_handler(ScopedHandler* handler)
{
    ScopedMap & variables = handler->get_variables();

    // Clean up
    SceneObject* tmp = variables.get_variable<SceneObject*>("temporary_object");
    variables.pop_variable<SceneObject*>("temporary_object");
    delete tmp;
    variables.pop_variable<SceneObject*>("current_object");
    variables.pop_variable<SceneObject *>("root_object");
}

void SceneFileLoader::scene_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap & attributes,
    ScopedHandler *handler)
{
    ScopedHandler::TagScope scope;
    scope.start_functors = global_start_handlers();
    handler->enter_scope(scope);
}


void SceneFileLoader::color_start_handler(
    const std::string & name,
    ScopedHandler::AttributeMap & attributes,
    ScopedHandler *handler)
{
    RGBColor color;

    color.r = atof(attributes["r"].c_str());
    color.g = atof(attributes["g"].c_str());
    color.b = atof(attributes["b"].c_str());

    // color scope - allow the same handlers as parent scope
    handler->duplicate_scope();
    handler->get_current_scope().end_functors.push_back(color_end_handler);

    ScopedMap & variables = handler->get_variables();
    variables.push_variable("color", color);
}

void SceneFileLoader::color_end_handler(const std::string & name,
                       ScopedHandler* handler)
{
    ScopedMap & variables = handler->get_variables();
    variables.pop_variable<RGBColor>("color");
}

void SceneFileLoader::layer_start_handler(
    const std::string & name,
    ScopedHandler::AttributeMap & attributes,
    ScopedHandler *handler)
{
    boost::uint8_t layer =
        boost::lexical_cast<boost::uint8_t>(attributes["number"]);

    // layer scope - allow the same handlers as parent scope
    handler->duplicate_scope();
    handler->get_current_scope().end_functors.push_back(layer_end_handler);

    ScopedMap & variables = handler->get_variables();
    variables.push_variable("layer", layer);

}

void SceneFileLoader::layer_end_handler(
    const std::string & name, ScopedHandler* handler)
{
    ScopedMap & variables = handler->get_variables();
    variables.pop_variable<boost::uint8_t>("layer");
}


void SceneFileLoader::external_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;
    scope.cdata_functor = &do_nothing;
    scope.unhandled_functor = &SceneFileLoader::external_start_handler;

    handler->enter_scope(scope);
}


void SceneFileLoader::include_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    assert( attributes.count("file") > 0 );

    ScopedMap &variables = handler->get_variables();

    assert( variables.exists<Path>("scene_file_path") );
    assert( variables.exists<boost::filesystem::path>("current_file_dir") );
    Path path = variables.get_variable<Path>("scene_file_path");
    path.prepend_search_dir(
        variables.get_variable<boost::filesystem::path>("current_file_dir"));
    path.set_path(boost::trim_copy(attributes["file"]));

    // Load file to be included
    assert( variables.exists<const Config *>("config") );
    std::vector<SceneObject *> objs = load_scene(
        *variables.get_variable<const Config *>("config"), path);

    SceneObject *parent = variables.get_variable<SceneObject *>(
        "current_object");
    std::map<SceneObject *, std::string> &orig_names =
        variables.get_variable<std::map<SceneObject *, std::string> >(
            "orig_names");

    try
    {
        for( size_t i = 0; i < objs.size(); ++i )
        {
            orig_names[objs[i]] = objs[i]->get_name();
            parent->attach(objs[i], AUTO_ENUMERATE_ON_CONFLICT);
        }
    }
    catch(...)
    {
        for( size_t i = 0; i < objs.size(); ++i )
            if( objs[i]->get_parent() == 0 )
                delete objs[i];

        throw;
    }



    ScopedHandler::TagScope scope;
    scope.start_functors["inject"] = &SceneFileLoader::inject_start_handler;
    scope.start_functors["external"] = &SceneFileLoader::external_start_handler;

    handler->enter_scope(scope);
}


bool SceneFileLoader::register_creator(
    const std::string &name,
    ScopedHandler::ElementStartFunctor handler)
{
    return global_start_handlers().insert(
        std::make_pair(name, handler)).second;
}


const ScopedHandler::ElementStartFunctor & SceneFileLoader::get_creator(
    const std::string & name) throw(std::runtime_error)
{
    ScopedHandler::StartFunctorMap::iterator it =
        global_start_handlers().find(name);

    if(it != global_start_handlers().end())
        return it->second;
    else
        throw std::runtime_error(
            (boost::format("Nonexistant start tag handler requested: %1%")
             % name).str());

}

const ScopedHandler::StartFunctorMap & SceneFileLoader::get_start_handlers()
{
    return global_start_handlers();
}

HandlerInformer::HandlerInformer(const std::string &name,
                                 ScopedHandler::ElementStartFunctor handler)
{
    SceneFileLoader::register_creator(name, handler);
}


void SceneFileLoader::inject_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    assert( attributes.count("at") > 0 );

    ScopedMap &variables = handler->get_variables();

    //
    // Find the current "current_object" and extract the injection point
    // from it and the "at" attribute.
    //
    SceneObject *prev_current_object = variables.get_variable<SceneObject *>(
        "current_object");

    std::string injection_path = boost::trim_copy(attributes["at"]);
    SceneObject *injection_point = 0;

    try
    {
        injection_point = prev_current_object->get_descendent(injection_path);
    }
    catch(...)
    {
        throw std::runtime_error("Injection point '" + injection_path + "' not found");
    }

    //
    // Override "current_object" with the object where the objects in the
    // inject block should be injected. Also set up a end_handler to pop
    // the overriden "current_object" parameter.
    //
    variables.push_variable("current_object", injection_point);


    ScopedHandler::TagScope scope;

    scope.end_functors.push_back(
        ScopedHandler::ElementEndFunctor(
            &SceneFileLoader::inject_end_handler));

    scope.start_functors = global_start_handlers();

    handler->enter_scope(scope);
}

void SceneFileLoader::inject_end_handler(
    const std::string &name,
    ScopedHandler *handler)
{
    // Restore the "current_object", which we overrode upon entering the
    // <inject> block (AKA in inject_start_handler)
    ScopedMap &variables = handler->get_variables();
    variables.pop_variable<SceneObject *>("current_object");
}
