/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2006.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>

#include "SceneTree.hh"
#include "SceneObject.hh"
#include "CameraObject.hh"
#include "GroupObject.hh"
#include "IDAllocator.hh"


using namespace peekabot;



// --------------------- TranslationTable implementation -----------------------

SceneTree::TranslationTable::TranslationTable(SceneObject *root) throw()
{
    on_child_attached(root);
}


void SceneTree::TranslationTable::on_child_attached(SceneObject *object) throw()
{
    add_rule(object);

    object->child_attached_signal().connect(
        boost::bind(&SceneTree::TranslationTable::on_child_attached, this, _1));

    object->detached_signal().connect(
        boost::bind(&SceneTree::TranslationTable::on_detached, this, object));

    for( SceneObject::ChildIterator it = object->begin();
         it != object->end(); ++it )
    {
        on_child_attached(*it);
    }
}


void SceneTree::TranslationTable::on_detached(SceneObject *object) throw()
{
    remove_rule(object);

    object->child_attached_signal().disconnect(
        boost::bind(&SceneTree::TranslationTable::on_child_attached, this, _1));

    object->detached_signal().disconnect(
        boost::bind(&SceneTree::TranslationTable::on_detached, this, object));

    for( SceneObject::ChildIterator it = object->begin();
         it != object->end(); ++it )
    {
        on_detached(*it);
    }
}


bool SceneTree::TranslationTable::add_rule(SceneObject *object)
    throw()
{
    TranslationMap::iterator it = m_id_map.find(object->get_object_id());

    if( it != m_id_map.end() )
        return true;

    m_id_map.insert(std::make_pair(object->get_object_id(), object));

    return false;
}


bool SceneTree::TranslationTable::remove_rule(SceneObject *object)
    throw()
{
    TranslationMap::iterator it = m_id_map.find(object->get_object_id());

    if( it == m_id_map.end() )
        return true;

    m_id_map.erase(it);

    return false;
}


SceneObject *SceneTree::TranslationTable::translate(
    ObjectID id) const throw()
{
    TranslationMap::const_iterator it = m_id_map.find(id);

    if( it != m_id_map.end() )
        return it->second;
    else
        return 0;
}



// ------------------------- SceneTree implementation -----------------------



SceneTree::SceneTree()
    : m_root(new GroupObject()),
      m_translation_table(m_root)
{
    SceneTreeMutex::scoped_lock lock(m_scene_mutex);

    m_root->set_name("root");

    CameraObject *default_cam = new CameraObject();
    default_cam->set_name("default_camera");
    m_root->attach(default_cam);
}

SceneTree::~SceneTree() throw()
{
    SceneTreeMutex::scoped_lock lock(m_scene_mutex);

    if( m_root )
        delete m_root;
}


SceneObject *SceneTree::get_root() throw()
{
    return m_root;
}

SceneObject *SceneTree::get_object(ObjectID id) throw()
{
    return m_translation_table.translate(id);
}


SceneObject *SceneTree::get_object(
    const std::string &path, SceneObject *path_parent) throw()
{
    std::vector<std::string> components;
    boost::split( components, path, boost::is_any_of("/") );

    if( components.empty() )
        components.push_back(path);

    SceneObject *obj = path_parent ? path_parent : m_root;

    try
    {
        for( std::size_t idx = 0; idx < components.size(); ++idx )
        {
            assert( obj != 0 );
            obj = obj->get_child(components[idx]);
        }
    }
    catch(...)
    {
        return 0;
    }

    return obj;
}
