/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include "ScopedHandler.hh"

using namespace peekabot;

ScopedHandler::ScopedHandler(HandlerFunctor start_handler,
                             HandlerFunctor end_handler,
                             HandlerFunctor failure_handler)
    : m_start_handler(start_handler), m_end_handler(end_handler),
      m_failure_handler(failure_handler)
{
}

void ScopedHandler::on_start_element(const std::string & tag_name,
                                     AttributeMap & attributes)
    throw(std::exception)
{
    if(m_scoped_tag_handlers.empty())
    {
        throw std::runtime_error(
            "ScopedHandler: Start of element encountered with empty scope "
            "stack: " + tag_name);
    }  

    // The current scope is searched for tag handlers
    TagScope scope = m_scoped_tag_handlers.top();
    
    // If  there is no tag handling functor for this tag, throw!
    if(scope.start_functors.find(tag_name) == scope.start_functors.end())
    {
        if( scope.unhandled_functor.empty() )
        {
            throw std::runtime_error(
                "ScopedHandler: Unhandled start element encountered with "
                "empty unhandled functor in the current scope");
        }

        scope.unhandled_functor(tag_name, attributes, this);
    }
    else
    {
        // We need to make sure that a new scope is entered properly.
        m_new_scope = false;
        // Call the tag handling functor
        scope.start_functors[tag_name](tag_name, attributes, this);

        // Now, if the start handler entered a new scope, this variable
        // will have been set to false.
        if(!m_new_scope)
            enter_scope();
    }
}


void ScopedHandler::on_end_element(const std::string & tag_name)
    throw(std::exception)
{
    if(m_scoped_tag_handlers.empty())
    {
        throw std::runtime_error(
            "ScopedHandler: End of element encountered with empty scope "
            "stack!\nElement: " + tag_name);
    }

    std::vector<ElementEndFunctor> end_functors = 
        m_scoped_tag_handlers.top().end_functors;
    for( size_t i = 0; i < end_functors.size(); ++i )
    {
        // Call the tag handling functor
        end_functors[i](tag_name, this);
    }

    // Leave scope...
    m_scoped_tag_handlers.pop();
}

void ScopedHandler::on_cdata(const std::string & cdata)
    throw(std::exception)
{
    if(m_scoped_tag_handlers.empty())
    {
        throw std::runtime_error(
            "ScopedHandler: Character data encountered with empty scope "
            "stack: " + cdata);
    }

    // The character data handler in the current scope is used.
    TagScope scope = m_scoped_tag_handlers.top();

    if(scope.cdata_functor.empty())
    {
        throw std::runtime_error(
            "ScopedHandler: Character data encountered with no cdata handler "
            "in scope: " + cdata);
    }
    
    // Call cdata handler
    scope.cdata_functor(cdata, this);
}

ScopedMap & ScopedHandler::get_variables()
{
    return m_scoped_variables;
}

void ScopedHandler::enter_scope(const TagScope & tag_handlers)
{
    m_scoped_tag_handlers.push(tag_handlers);

    // Remember that we entered a new scope...
    m_new_scope = true;
}

void ScopedHandler::enter_scope()
{
    // Push an empty scope.
    m_scoped_tag_handlers.push(TagScope());

    // Remember that we entered a new scope...
    m_new_scope = true;
}

void ScopedHandler::duplicate_scope()
{
    // Duplicate the topmost scope 
    m_scoped_tag_handlers.push(m_scoped_tag_handlers.top());

    // Remember that we entered a new scope...
    m_new_scope = true;
}

void ScopedHandler::on_start_document() throw(std::exception)
{
    m_start_handler(this);
}
        
void ScopedHandler::on_end_document() throw(std::exception)
{
    m_end_handler(this);
}

void ScopedHandler::on_failure() throw(std::exception)
{
    m_failure_handler(this);
}

bool ScopedHandler::add_start_handler(const std::string &name,
                                      ElementStartFunctor handler)
{
    TagScope & scope = m_scoped_tag_handlers.top();
    
    return scope.start_functors.insert(std::make_pair(name, handler)).second;
}

void ScopedHandler::set_start_document_handler(HandlerFunctor start_handler)
{
    m_start_handler = start_handler;
}

void ScopedHandler::set_end_document_handler(HandlerFunctor end_handler)
{
    m_end_handler = end_handler;
}


void ScopedHandler::set_failure_handler(HandlerFunctor failure_handler)
{
    m_failure_handler = failure_handler;
}

ScopedHandler::TagScope &ScopedHandler::get_current_scope() 
    throw(std::runtime_error)
{
    if( m_scoped_tag_handlers.empty() )
        throw std::runtime_error(
            "ScopedHandler: Current scope requested, but scope stack is empty");

    return m_scoped_tag_handlers.top();
}
