/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_SCOPED_HANDLER_HH
#define __PEEKABOT_SCOPED_HANDLER_HH

#include <map>
#include <vector>
#include <string>
#include <stdexcept>
#include <boost/function.hpp>

#include "XMLHandler.hh"
#include "ScopedMap.hh"

namespace peekabot
{
    /**
     * \internal
     *
     * \brief XML input handler class that organizes handler methods by
     * scopes corresponding to the hierarchical levels in the XML document.
     *
     * Upon creation, the constructor is supplied with a document start and
     * end handler. The start handler is called immediately when the document
     * is opened, before any elements are parsed. It is the start handler's
     * task to define the top-level scope of the document by creating a
     * \c TagScope with a set of tag handling function objects (i.e. an \c
     * ElementFunctorMap and optionally a \c CharacterFunctor) that will
     * receive any top-level elements in the document. It is up to these
     * function objects to register new handlers for any element that may be
     * found within these first elements.
     *
     * The end handler is called when the parser reaches the end of the
     * document, but only if it has successfully parsed it, i.e. no errors
     * occurred. The end handler is optional, but the user may want to use it
     * to clean up any heap-allocated variables, or perhaps add parsed objects
     * to the scene.
     */
    class ScopedHandler : public XMLHandler
    {
    public:
        /** \brief A function object for handling start of element-tags. The
         * first argument is the name of the tag, the second is a list of
         * attributes found in the tag.
         */
        typedef boost::function<
            void (const std::string&, AttributeMap&,
                  ScopedHandler* handler)> ElementStartFunctor;
        /** \brief A function object for handling character data found in
         * an element. The argument supplied is the string of characters
         * found within the element.
         */
        typedef boost::function<
            void (const std::string&, ScopedHandler* handler)> CharacterFunctor;
        
        /** \brief A function object for handling end of element-tags. The
         * argument supplied is the name of the tag.
         */
        typedef boost::function<
            void (const std::string&, ScopedHandler* handler)> ElementEndFunctor;

        /** \brief Generic typedef for functors taking only a ScopedMap pointer
            argument.
        */
        typedef boost::function<void (ScopedHandler* handler)> HandlerFunctor;

        typedef ElementStartFunctor UnhandledElementFunctor;
        
        /** \brief A pair of corresponding start and end tags for an element.
         */
        typedef std::pair<ElementStartFunctor, ElementEndFunctor> StartEndPair;

        /** \brief A map associating element names with their start functors.
         */
        typedef std::map<std::string, ElementStartFunctor> StartFunctorMap;
    
        /**
         * \internal
         *
         * \brief Holds the cdata and tag handlers that are valid within
         * the current scope.
         *
         * For every element that may occur within this scope, there must
         * be a corresponding start functor defined. The \c start_functors
         * map holds these start functors, which are indexed by the name
         * of the element they handle.
         *
         * If character data may occur within this scope, there must also
         * be a \c cdata_functor defined.
         */
        struct TagScope
        {
            TagScope()
                : unhandled_functor(&default_unhandled) {}

            /** \brief Functor handling any character data encountered within
             * the current scope.
             */
            CharacterFunctor cdata_functor;

            /** \brief Map holding the start functors for the elements that are
             * allowed in the current scope.
             */
            StartFunctorMap start_functors;

            /** \brief Functors called at the end of the element.
             *
             * By default, it's a empty.
             */
            std::vector<ElementEndFunctor> end_functors;

            /** \brief Functor that is called when an unhandled start element is
             * encountered.
             *
             * By default, it's set to a function that throws an error.
             */
            UnhandledElementFunctor unhandled_functor;

        private:
            /** \brief Default unhandled-handler.
             */
            static void default_unhandled(
                const std::string &tag_name,
                AttributeMap &,
                ScopedHandler *handler) throw(std::runtime_error)
            {
                throw std::runtime_error(
                    "Unhandled start tag encountered: " + tag_name);
            }
        };

        /** \brief Create a new \c ScopedHandler with the the specified
         * document start and end handlers.
         */
        ScopedHandler(HandlerFunctor start_handler = do_nothing,
                      HandlerFunctor end_handler = do_nothing,
                      HandlerFunctor failure_handler = do_nothing);

        /** \brief Called by \c XMLParser when a new XML element is encountered.
         *
         * This method will call the corresponding \c ElementStartFunctor in the
         * top \c TagScope.
         *
         * \param tag_name The name of the tag encountered.
         * \param attributes A list of attribute name-value pairs.
         */
        virtual void on_start_element(const std::string & tag_name,
                                      AttributeMap & attributes)
            throw(std::exception);

        /** \brief Called by \c XMLParser when the end of an XML element is
         * reached.
         *
         * This method will call the topmost end handler in the stack and
         * leave the current scope.
         *
         * \param tag_name The name of the tag encountered.
         */
        virtual void on_end_element(const std::string & tag_name)
            throw(std::exception);
        
        /** \brief Called by \c XMLParser when characted data is encountered in
         * an element.
         *
         * This method will call the \c CharacterFunctor of the top \c TagScope.
         *
         * \param cdata The string of characters encountered.
         */
        virtual void on_cdata(const std::string & cdata)
            throw(std::exception);
        
        /** \brief Called by \c XMLParser at the beginning of a document.
         *
         * This method will call the \c HandlerFunctor specified with the
         * \c set_start_document_handler() method.
         */
        virtual void on_start_document() throw(std::exception);


        /** \brief Called by \c XMLParser at the end of a document.
         *
         * This method will call the \c HandlerFunctor specified with the
         * \c set_end_document_handler() method.
         */      
        virtual void on_end_document() throw(std::exception);

        /** \brief Calls the failure handler to do clean up operations.
         */        
        virtual void on_failure() throw(std::exception);

        /** \brief Returns a reference to the \c ScopedMap containing this
         * handler's variables.
         */
        ScopedMap & get_variables();

        /** \brief Enters a new scope, locally overriding any tag handlers.
         */
        void enter_scope(const TagScope & tag_handlers);

        /** \brief Enters a new scope, without any start handlers.
         */
        void enter_scope();

        /** \brief Enters a new scope, allowing the same handlers as the
         * current scope (the topmost scope on the stack is duplicated).
         */
        void duplicate_scope();

        /** \brief Adds a new start tag handler in the current scope.
         */
        bool add_start_handler(const std::string &name,
                               ElementStartFunctor start_handler);

        /** \brief Specifies the \c HandlerFunctor to be called when the start
         * of a document is encountered.
         */
        void set_start_document_handler(HandlerFunctor start_handler);

        /** \brief Specifies the \c HandlerFunctor to be called when the end
         * of a document is encountered.
         */
        void set_end_document_handler(HandlerFunctor end_handler);

        /** \brief Specifies the \c HandlerFunctor to be called when the
         * document could not be parsed correctly.
         */
        void set_failure_handler(HandlerFunctor failure_handler);

        TagScope &get_current_scope() throw(std::runtime_error);


    private:
        /** \brief Dummy function doing nothing.
         */
        static void do_nothing(ScopedHandler* handler) {}

        std::stack<TagScope> m_scoped_tag_handlers;
        ScopedMap m_scoped_variables;

        HandlerFunctor m_start_handler;
        HandlerFunctor m_end_handler;
        HandlerFunctor m_failure_handler;

        bool m_new_scope;
    };
    
}
#endif //__PEEKABOT_SCOPED_HANDLER_HH
