/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_SCOPED_MAP_HH
#define __PEEKABOT_SCOPED_MAP_HH

#include <stdexcept>
#include <map>
#include <stack>
#include <boost/any.hpp>

namespace peekabot
{
    /**
     * \internal
     *
     * \brief Map for storing dynamic variables of arbitrary type, whose value
     * can be overridden in local scopes.
     *
     * The concept of a "scope" is here maintained by associating each variable
     * name with a stack of the variable's type. The value of the variable in
     * the current scope is then the topmost value in that stack, and is
     * retrieved by calling the \c get_variable() method.
     *
     * Entering and leaving a "scope" is accomplished by calling the
     * \c push_variable() and \c pop_variable() methods respectively.
     */
    class ScopedMap
    {
    public:
        /** \brief Gets the value of the specified variable in the current scope.
         *
         * \param variable_name The name of the variable to retrieve.
         *
         * \exception boost::bad_any_cast Cast when the template parameter type
         * doesn't match the type of the variable requested.
         *
         * \exception std::exception Cast when the requested variable doesn't
         * exist.
         */
        template<class T> T & get_variable(const std::string & variable_name)
            throw(boost::bad_any_cast, std::runtime_error)
        {
            // Verify that the variable exists
            if(m_any_map.find(variable_name) == m_any_map.end())
            {
                std::string error = "ScopedMap: variable \"";
                error += variable_name + "\" not found!";
                throw(std::runtime_error(error));
            }
            
            // Fetch the variable's scope stack
            boost::any & any_stack = m_any_map[variable_name];
            // May throw a bad_any_cast exception:
            std::stack<T> & value_stack = boost::any_cast<std::stack<T>&>(any_stack);
            
            return value_stack.top();
        }

        /** \brief Pushes a new value for the variable onto the scope stack.
         *
         * \param variable_name The name of the variable to override.
         *
         * \exception boost::bad_any_cast Cast when the template parameter type
         * doesn't match the type of the variable requested.
         *
         */
        template<class T> void push_variable(const std::string & variable_name,
                                             const T& value)
            throw(boost::bad_any_cast)
        {
            // If the variable exists, push it to the top of the stack
            if(m_any_map.find(variable_name) != m_any_map.end())
            {
                boost::any & any_stack = m_any_map[variable_name];
                std::stack<T> & scope_stack = boost::any_cast<std::stack<T>&>(any_stack);
                
                scope_stack.push(value);                
            }
            else
            {
                std::stack<T> s;
                s.push(value);
                m_any_map[variable_name] = s;

            }

        }

        /** \brief Pops the topmost value of a variable from the scope stack.
         *
         * \param variable_name The name of the variable to retrieve.
         *
         * \exception boost::bad_any_cast Cast when the template parameter type
         * doesn't match the type of the variable requested.
         *
         * \exception std::exception Cast when the requested variable doesn't
         * exist.
         */
        template<class T> void pop_variable(const std::string & variable_name)
            throw(boost::bad_any_cast, std::runtime_error)
        {
            // Verify that the variable exists
            if(m_any_map.find(variable_name) == m_any_map.end())
            {
                std::string error = "ScopedMap: variable \"";
                error += variable_name + "\" not found!";
                throw(std::runtime_error(error));
            }
            
            // Fetch the variable's scope stack
            boost::any & any_stack = m_any_map[variable_name];
            // May throw a bad_any_cast exception:
            std::stack<T> & value_stack = boost::any_cast<std::stack<T>&>(any_stack);
            
            value_stack.pop();
        }


        /** \brief Checks to see if there exists a variable with the specified
         * name and parameterized type.
         */
        template<class T> bool exists(const std::string & variable_name)
        {
            // Check if there exists a variable with the specified name...
            if(m_any_map.find(variable_name) == m_any_map.end())
                return false;
            
            // The variable exists, but is it of the right type?
            try
            {
                // Fetch the variable's scope stack
                boost::any & any_stack = m_any_map[variable_name];
                // May throw a bad_any_cast exception:
                std::stack<T> & value_stack = boost::any_cast<std::stack<T>&>(any_stack);
                
                // If the stack is empty, there is no value for the variable, it has gone
                // out of scope, i.e. it doesn't exist.
                if(value_stack.empty())
                    return false;
                else // If it's non-empty, the variable exists and has a value.
                    return true;
            }
            catch(boost::bad_any_cast & error)
            {
                // If we caught this exception, it means that the variable existed, but
                // it was of the wrong type.
                return false;
            }
        }
        
        
    private:
        std::map<std::string, boost::any> m_any_map;
    };
    
}

#endif //__PEEKABOT_SCOPED_MAP_HH
