/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/bind.hpp>
#include <stdexcept>
#include <sstream>

#include "VertexObject.hh"
#include "SceneObject.hh"
#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "PropKeys.hh"
#include "props/Uint32PropBase.hh"
#include "props/EnumPropBase.hh"


using namespace peekabot;



VertexObject::VertexObject(boost::uint32_t multiple_of) throw()
    : m_multiple_of(multiple_of),
      m_max_vertices(0),
      m_overflow_policy(VERTEX_OVERFLOW_TRUNCATE_HALF)
{
}


VertexObject::VertexObject(ScopedHandler *handler, boost::uint32_t multiple_of)
    : m_multiple_of(multiple_of),
      m_max_vertices(0),
      m_overflow_policy(VERTEX_OVERFLOW_TRUNCATE_HALF)
{
    boost::function<void (const std::string&, ScopedHandler*)> fn =
        boost::bind(&VertexObject::vertices_end_handler, this, _1, _2);

    // Register the vertices tag start handler
    handler->add_start_handler(
        "vertices",
        boost::bind(&VertexObject::vertices_start_handler,
                    this, fn, _1, _2, _3));
}


void VertexObject::set_vertices(const Vertices &vertices)
{
    if( vertices.size() % m_multiple_of != 0 )
        throw std::length_error("Invalid number of vertices");

    m_vertices = vertices;
    handle_overflow();
    m_vertices_set_signal();
}


void VertexObject::add_vertices(const Vertices &vertices)
{
    std::size_t n = vertices.size() + get_vertex_count();

    if( n % m_multiple_of != 0 )
        throw std::length_error("Invalid number of vertices");

    m_vertices.insert(m_vertices.end(), vertices.begin(), vertices.end());
    if( handle_overflow() )
    {
        // Vertices overflowed, so we can't just emit the added vertices
        m_vertices_set_signal();
    }
    else
    {
        m_vertices_added_signal(vertices, 0);
    }
}


const VertexObject::Vertices &VertexObject::get_vertices() const throw()
{
    return m_vertices;
}


void VertexObject::set_colored_vertices(
    const Vertices &vertices, const VertexColors &colors)
{
    if( vertices.size() % m_multiple_of != 0 )
        throw std::length_error("Invalid number of vertices");

    assert( 3*vertices.size() == colors.size() );
    m_vertices = vertices;
    m_vertex_colors = colors;

    m_vertices_set_signal();
}


void VertexObject::add_colored_vertices(
    const Vertices &vertices, const VertexColors &colors)
{
    if( !has_vertex_colors() )
        throw std::logic_error("Object does not have vertex colors");

    std::size_t n = vertices.size() + get_vertex_count();

    if( n % m_multiple_of != 0 )
        throw std::length_error("Invalid number of vertices");

    m_vertices.insert(m_vertices.end(), vertices.begin(), vertices.end());
    m_vertex_colors.insert(m_vertex_colors.end(), colors.begin(), colors.end());

    if( handle_overflow() )
    {
        // Vertices overflowed, so we can't just emit the added vertices
        m_vertices_set_signal();
    }
    else
    {
        m_vertices_added_signal(vertices, &colors);
    }
}


const VertexObject::VertexColors &VertexObject::get_vertex_colors() const
{
    if( !has_vertex_colors() )
        throw std::logic_error("Object does not have vertex colors");
    return m_vertex_colors;
}


bool VertexObject::has_vertex_colors() const
{
    return !m_vertex_colors.empty();
}


std::size_t VertexObject::get_vertex_count() const throw()
{
    return m_vertices.size();
}


void VertexObject::set_overflow_policy(VertexOverflowPolicy policy)
{
    m_overflow_policy = policy;
    m_overflow_policy_set_signal();
    if( handle_overflow() )
        m_vertices_set_signal();
}


VertexOverflowPolicy VertexObject::get_overflow_policy() const
{
    return m_overflow_policy;
}


void VertexObject::set_max_vertices(boost::uint32_t n)
{
    m_max_vertices = n;
    m_max_vertices_set_signal();
    if( handle_overflow() )
        m_vertices_set_signal();
}


boost::uint32_t VertexObject::get_max_vertices() const
{
    return m_max_vertices;
}


bool VertexObject::handle_overflow()
{
    if( m_max_vertices == 0 )
    {
        return false;
    }
    else if( m_vertices.size() <= m_max_vertices )
    {
        return false;
    }
    else
    {
        if( m_overflow_policy == VERTEX_OVERFLOW_CLEAR )
        {
            m_vertices.clear();
            if( has_vertex_colors() )
                m_vertex_colors.clear();
        }
        else
        {
            // The number of vertices after truncation
            boost::int32_t n;

            if( m_overflow_policy == VERTEX_OVERFLOW_TRUNCATE )
                n = m_max_vertices - m_max_vertices % m_multiple_of;
            else if( m_overflow_policy == VERTEX_OVERFLOW_TRUNCATE_HALF )
                n = m_max_vertices/2 - (m_max_vertices/2) % m_multiple_of;
            else
                assert( false );

            if( n < 0 )
                n = 0;

            assert( (std::size_t)n <= m_vertices.size() );
            std::size_t off = m_vertices.size() - n;
            m_vertices = Vertices(m_vertices.begin()+off, m_vertices.end());

            if( has_vertex_colors() )
            {
                assert( (std::size_t)3*n <= m_vertex_colors.size() );
                std::size_t off = m_vertex_colors.size() - 3*n;
                m_vertex_colors = VertexColors(
                    m_vertex_colors.begin()+off, m_vertex_colors.end());
            }
        }

        return true;
    }
}


PropMap &VertexObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
    }

    return *s_prop_adapters;
}


void VertexObject::create_prop_adapters(PropMap &adapters)
{
    class OverflowPolicyAdapter : public EnumPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            VertexObject *p = dynamic_cast<VertexObject *>(obj);
            assert( p );
            p->set_overflow_policy(any_cast<VertexOverflowPolicy>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const VertexObject *p = dynamic_cast<const VertexObject *>(obj);
            assert( p );
            return Any(p->get_overflow_policy());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            VertexObject *p = dynamic_cast<VertexObject *>(obj);
            assert( p );
            return p->overflow_policy_set_signal();
        }

        virtual void get_enum_values(std::vector<std::string> &v) const
        {
            v.push_back("Clear");
            v.push_back("Truncate");
            v.push_back("Truncate half");
        }

        virtual void set_from_text(const std::string &text, SceneObject *obj)
        {
            VertexObject *p = dynamic_cast<VertexObject *>(obj);
            assert( p );

            if( text == "Clear" )
                p->set_overflow_policy(VERTEX_OVERFLOW_CLEAR);
            else if( text == "Truncate" )
                p->set_overflow_policy(VERTEX_OVERFLOW_TRUNCATE);
            else if( text == "Truncate half" )
                p->set_overflow_policy(VERTEX_OVERFLOW_TRUNCATE_HALF);
            else
                assert( false );
        }

        virtual std::string get_as_text(const SceneObject *obj) const
        {
            const VertexObject *p = dynamic_cast<const VertexObject *>(obj);
            assert( p );

            if( p->get_overflow_policy() == VERTEX_OVERFLOW_CLEAR )
                return "Clear";
            else if( p->get_overflow_policy() == VERTEX_OVERFLOW_TRUNCATE )
                return "Truncate";
            else if( p->get_overflow_policy() == VERTEX_OVERFLOW_TRUNCATE_HALF )
                return "Truncate half";
            else
                assert( false );
        }
    };

    class MaxVerticesAdapter : public Uint32PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            VertexObject *p = dynamic_cast<VertexObject *>(obj);
            assert( p );
            p->set_max_vertices(any_cast<boost::uint32_t>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const VertexObject *p = dynamic_cast<const VertexObject *>(obj);
            assert( p );
            return Any(p->get_max_vertices());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            VertexObject *p = dynamic_cast<VertexObject *>(obj);
            assert( p );
            return p->max_vertices_set_signal();
        }
    };

    adapters.insert(
        PropMap::value_type(VERTEX_OVERFLOW_POLICY_PROP, new OverflowPolicyAdapter));

    adapters.insert(
        PropMap::value_type(MAX_VERTICES_PROP, new MaxVerticesAdapter));
}


// ---------------------- XML handler functions ----------------------


void VertexObject::vertices_start_handler(
    boost::function<void (const std::string&, ScopedHandler*)> end_handler,
    const std::string tag_name,
    XMLHandler::AttributeMap & attributes,
    ScopedHandler* handler)
{
    ScopedHandler::TagScope vertices_scope;

    vertices_scope.start_functors["vertex"] = boost::bind(
        &VertexObject::vertex_start_handler, _1, _2, _3);

    vertices_scope.end_functors.push_back(end_handler);

    handler->get_variables().push_variable<Vertices>("vertices", Vertices());

    handler->enter_scope(vertices_scope);
}


void VertexObject::vertices_end_handler(const std::string &name,
                                        ScopedHandler *handler)
    throw(std::runtime_error, std::length_error)
{
    ScopedMap &variables = handler->get_variables();
    assert( variables.exists<Vertices>("vertices") );
    Vertices &vertices = variables.get_variable<Vertices>("vertices");
    set_vertices(vertices);
    variables.pop_variable<Vertices>("vertices");
}


void VertexObject::vertex_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler)
{
    ScopedMap & variables = handler->get_variables();

    // Read coordinate information
    std::stringstream s(cdata);
    Eigen::Vector3f vec;
    s >> vec(0) >> vec(1) >> vec(2);

    variables.get_variable<Vertices>("vertices").push_back(vec);
}


void VertexObject::vertex_start_handler(
    const std::string & tag_name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    ScopedHandler::TagScope vertex_scope;

    vertex_scope.cdata_functor = boost::bind(
        &vertex_cdata_handler, _1, _2);

    handler->enter_scope(vertex_scope);
}
