/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_VERTEX_OBJECT_HH_INCLUDED
#define PEEKABOT_VERTEX_OBJECT_HH_INCLUDED


#include <vector>
#include <boost/cstdint.hpp>
#include <boost/signals2.hpp>
#include <Eigen/Core>

#include "Types.hh"
#include "XMLHandler.hh"
#include "PropMap.hh"

namespace peekabot
{
    class ScopedHandler;

    class VertexObject
    {
    public:
        typedef std::vector<Eigen::Vector3f> Vertices;
        typedef std::vector<boost::uint8_t> VertexColors;

        VertexObject(boost::uint32_t multiple_of = 1) throw();

        /**
         * \brief Construct a vertex object using an XML document handler.
         */
        VertexObject(ScopedHandler *handler, boost::uint32_t multiple_of = 1);

        virtual ~VertexObject() {}

        void set_vertices(const Vertices &vertices);

        void add_vertices(const Vertices &vertices);

        const Vertices &get_vertices() const throw();

        void set_colored_vertices(const Vertices &vertices, const VertexColors &colors);

        void add_colored_vertices(const Vertices &vertices, const VertexColors &colors);

        const VertexColors &get_vertex_colors() const;

        bool has_vertex_colors() const;

        std::size_t get_vertex_count() const throw();

        void set_overflow_policy(VertexOverflowPolicy policy);

        VertexOverflowPolicy get_overflow_policy() const;

        void set_max_vertices(boost::uint32_t n);

        boost::uint32_t get_max_vertices() const;

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> VerticesSetSignal;

        typedef boost::signals2::signal<void (
            const Vertices &new_vertices,
            const VertexColors *new_colors)> VerticesAddedSignal;

        typedef boost::signals2::signal<void ()> OverflowPolicySetSignal;

        typedef boost::signals2::signal<void ()> MaxVerticesSetSignal;

        inline VerticesSetSignal &vertices_set_signal() const
        {
            return m_vertices_set_signal;
        }

        inline VerticesAddedSignal &vertices_added_signal() const
        {
            return m_vertices_added_signal;
        }

        inline OverflowPolicySetSignal &overflow_policy_set_signal() const
        {
            return m_overflow_policy_set_signal;
        }

        inline MaxVerticesSetSignal &max_vertices_set_signal() const
        {
            return m_max_vertices_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        bool handle_overflow();

        /// \name XML handler methods
        /// @{

        void vertices_start_handler(
            boost::function<void (const std::string&, ScopedHandler*)> end_handler,
            const std::string tag_name,
            XMLHandler::AttributeMap & attributes,
            ScopedHandler* handler);

        void vertices_end_handler(
            const std::string &name, ScopedHandler *handler)
            throw(std::runtime_error, std::length_error);


        static void vertex_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler);


        static void vertex_start_handler(
            const std::string & tag_name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler);

        /// @}

    private:
        const boost::uint32_t m_multiple_of;

        boost::uint32_t m_max_vertices;
        VertexOverflowPolicy m_overflow_policy;

        Vertices m_vertices;
        VertexColors m_vertex_colors;

        mutable VerticesSetSignal m_vertices_set_signal;
        mutable VerticesAddedSignal m_vertices_added_signal;
        mutable OverflowPolicySetSignal m_overflow_policy_set_signal;
        mutable MaxVerticesSetSignal m_max_vertices_set_signal;
    };
}


#endif // PEEKABOT_VERTEX_OBJECT_HH_INCLUDED
