/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_XML_HANDLER_HH
#define __PEEKABOT_XML_HANDLER_HH

#include <string>
#include <map>

namespace peekabot
{
    /**
     * \internal
     *
     * \brief Base class for handling SAX2 XML parsing events.
     *
     * In the base form, all methods in this class are set to do
     * nothing upon receiving events. Subclasses should implement
     * handler methods for each type of event as necessary.
     */
    class XMLHandler
    {
    public:
        /** \brief A map of name-value attribute pairs.
         */
        typedef std::map<std::string, std::string> AttributeMap;

        virtual ~XMLHandler() {}

        /** \brief Called by \c XMLParser when a new XML element is encountered.
         *
         * \param tag_name The name of the tag encountered.
         * \param attributes A list of attribute name-value pairs.
         */
        virtual void on_start_element(const std::string & tag_name,
                                      AttributeMap & attributes)
            throw(std::exception) {}
        
        /** \brief Called by \c XMLParser when the end of an XML element is
         * reached.
         *
         * \param tag_name The name of the tag encountered.
         */
        virtual void on_end_element(const std::string & tag_name)
            throw(std::exception) {}
        
        /** \brief Called by \c XMLParser when characted data is encountered in
         * an element.
         *
         * \param cdata The string of characters encountered.
         */
        virtual void on_cdata(const std::string & cdata)
            throw(std::exception) {}
        
        /** \brief Called by \c XMLParser at the beginning of a document.
         */
        virtual void on_start_document() throw(std::exception) {}

        /** \brief Called by \c XMLParser at the end of a document.
         */        
        virtual void on_end_document() throw(std::exception) {}   

        /** \brief Called by \c XMLParser if the parsing fails.
         */        
        virtual void on_failure() throw(std::exception) {}
    };

}

#endif //__PEEKABOT_XML_HANDLER_HH
