/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <sstream>
#include <boost/format.hpp>

#include "Bundle.hh"
#include "../serialization/Containers.hh"
#include "../serialization/SmartPtrs.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#else
#  include "../client/ClientExecutionContext.hh"
#endif


using namespace peekabot;


Bundle::Bundle() throw()
{
}


Bundle::Bundle(const Bundle &bundle) throw()
{
    for( std::vector<boost::shared_ptr<Action> >::const_iterator it =
             bundle.m_actions.begin();
         it != bundle.m_actions.end(); ++it )
    {
        add_action(boost::shared_ptr<Action>((*it)->clone()));
    }
}


Bundle::~Bundle() throw()
{
}


Action *Bundle::clone() const
{
    return new Bundle(*this);
}


void Bundle::execute(
    ServerExecutionContext *context) 
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    size_t n_errors = 0, action_no = 0;
    std::stringstream error_log;

    for( std::vector<boost::shared_ptr<Action> >::const_iterator it =
             m_actions.begin();
         it != m_actions.end(); ++it )
    {
        ++action_no;
        try
        {
            (*it)->execute(context);
        }
        catch(std::exception &e)
        {
            ++n_errors;
            error_log << "\n  Action " << action_no << ": " << e.what();
        }
        catch(...)
        {
            ++n_errors;
            error_log << "\n  Action " << action_no << ": Unknown exception";
        }
    }

    if( n_errors > 0 )
        throw std::runtime_error(
            (boost::format(
                "Bundle failed - %1% out of %2% actions failed. Errors:%3%") 
             % n_errors % m_actions.size() % error_log.str() ).str() );
#endif
}


void Bundle::execute(
    ClientExecutionContext *context)
    const throw(std::exception)
{
#ifdef __PEEKABOT_CLIENT
    size_t n_errors = 0;

    for( std::vector<boost::shared_ptr<Action> >::const_iterator it =
             m_actions.begin();
         it != m_actions.end(); ++it )
    {
        try
        {
            (*it)->execute(context);
        }
        catch(...)
        {
            ++n_errors;
        }
    }

    if( n_errors > 0 )
        throw std::runtime_error(
            (boost::format("Bundle failed - %1% out of %2% actions failed.") 
             % n_errors % m_actions.size() ).str() );
#endif
}


void Bundle::add_action(boost::shared_ptr<Action> action) throw()
{
    assert(action.get() != this);
    m_actions.push_back(action);
}

void Bundle::save(SerializationInterface &ar) const
{
    ar << m_actions;
}

void Bundle::load(DeserializationInterface &ar)
{
    ar >> m_actions;
}
