/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "GetTransformation.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "ActionResultContainer.hh"
#  include "../serialization/Types.hh"
#  include "../SceneObject.hh"
#endif

#include <stdexcept>


using namespace peekabot;


GetTransformation::GetTransformation() throw()
{
}


GetTransformation::GetTransformation(
    boost::uint32_t request_id,
    ObjectID target,
    CoordinateSystem coord_sys) throw()
    : m_request_id(request_id),
      m_target(target),
      m_coord_sys(coord_sys)
{
}


GetTransformation::GetTransformation(const GetTransformation &x) throw()
    : m_request_id(x.m_request_id),
      m_target(x.m_target),
      m_coord_sys(x.m_coord_sys)
{
}


GetTransformation::~GetTransformation() throw()
{
}


Action *GetTransformation::clone() const
{
    return new GetTransformation(*this);
}


void GetTransformation::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Failed to get transformation, target object not found.");

    Eigen::Transform3f M;

    switch( m_coord_sys )
    {
        case PARENT_COORDINATES:
            M = ptr->get_transformation();
            break;

        case WORLD_COORDINATES:
            M = ptr->get_mtow();
            break;

        default:
            throw std::runtime_error(
                "Failed to get transformation: unsupported coordinate system");
            break;
    }

    Transformation res;
    res.x   = Vector3(M(0,0), M(1,0), M(2,0));
    res.y   = Vector3(M(0,1), M(1,1), M(2,1));
    res.z   = Vector3(M(0,2), M(1,2), M(2,2));
    res.pos = Vector3(M(0,3), M(1,3), M(2,3));

    context->post_response(new ActionResultContainer(m_request_id, Any(res)));
#endif
}


void GetTransformation::save(SerializationInterface &ar) const
{
    ar << m_request_id << m_target << m_coord_sys;
}

void GetTransformation::load(DeserializationInterface &ar)
{
    ar >> m_request_id >> m_target >> m_coord_sys;
}
