/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <boost/math/fpclassify.hpp>
#include <Eigen/Core>
#include <Eigen/LU>

#include "Rotate.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


Rotate::Rotate() throw()
{
}


Rotate::Rotate(
    ObjectID target, float rad,
    float axis_x, float axis_y, float axis_z,
    CoordinateSystem axis_coord_sys,
    float pivot_x, float pivot_y, float pivot_z,
    CoordinateSystem pivot_coord_sys) throw()
    : m_target(target), m_rad(rad),
      m_axis_x(axis_x), m_axis_y(axis_y), m_axis_z(axis_z),
      m_axis_coord_sys(axis_coord_sys),
      m_pivot_x(pivot_x), m_pivot_y(pivot_y), m_pivot_z(pivot_z),
      m_pivot_coord_sys(pivot_coord_sys)
{
}


Action *Rotate::clone() const
{
    return new Rotate(
        m_target, m_rad,
        m_axis_x, m_axis_y, m_axis_z, m_axis_coord_sys,
        m_pivot_x, m_pivot_y, m_pivot_z, m_pivot_coord_sys);
}


void Rotate::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error("Target object not found.");

    // Check for infinity and NaN
    if( !boost::math::isfinite(m_axis_x)  ||
        !boost::math::isfinite(m_axis_y)  ||
        !boost::math::isfinite(m_axis_z)  ||
        !boost::math::isfinite(m_pivot_x) ||
        !boost::math::isfinite(m_pivot_y) ||
        !boost::math::isfinite(m_pivot_z) )
        throw std::logic_error(
            "rotate failed: the rotational axis and pivot "
            "cannot contain infinity or NaN");

    // Check for zero-length axis vector
    const float axis_norm_sq =
        m_axis_x*m_axis_x + m_axis_y*m_axis_y + m_axis_z*m_axis_z;
    if( axis_norm_sq == 0 )
        throw std::logic_error(
            "rotate failed: the rotational axis must be non-zero");

    // We do the rotations in parent coordinates, so we need to transform the
    // pivot and axis to that coordinate system if they aren't already in it.
    //
    // Also, normalize the rotational axis.
    const float axis_norm = sqrt(axis_norm_sq);
    Eigen::Vector3f axis_p(
        m_axis_x/axis_norm, m_axis_y/axis_norm, m_axis_z/axis_norm);

    switch( m_axis_coord_sys )
    {
        case LOCAL_COORDINATES:
            axis_p = ptr->get_transformation().linear() * axis_p;
            break;

        case PARENT_COORDINATES:
            // Nothing needs to be done here
            break;

        case WORLD_COORDINATES:
            axis_p = Eigen::Transform3f(
                ptr->get_parent_mtow().inverse(Eigen::Isometry)).linear() *
                axis_p;
            break;

        default:
            throw std::runtime_error(
                "rotate failed: unsupported coordinate system");
            break;
    }

    Eigen::Vector3f piv_p(m_pivot_x, m_pivot_y, m_pivot_z);

    switch( m_pivot_coord_sys )
    {
        case LOCAL_COORDINATES:
            piv_p = ptr->get_transformation() * piv_p;
            break;

        case PARENT_COORDINATES:
            // Nothing needs to be done here
            break;

        case WORLD_COORDINATES:
            piv_p = Eigen::Transform3f(
                ptr->get_parent_mtow().inverse(Eigen::Isometry)) * piv_p;
            break;

        default:
            throw std::runtime_error(
                "rotate failed: unsupported coordinate system");
            break;
    }

    // New transformation: M' = T_piv * R * inv(T_piv) * M
    ptr->set_transformation(
        Eigen::Translation3f(piv_p(0), piv_p(1), piv_p(2)) *
        Eigen::AngleAxisf(m_rad, axis_p) *
        Eigen::Translation3f(-piv_p(0), -piv_p(1), -piv_p(2)) *
        ptr->get_transformation());
#endif
}


void Rotate::save(SerializationInterface &ar) const
{
    ar << m_target << m_rad
       << m_axis_x << m_axis_y << m_axis_z
       << m_axis_coord_sys
       << m_pivot_x << m_pivot_y << m_pivot_z
       << m_pivot_coord_sys;
}

void Rotate::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_rad
       >> m_axis_x >> m_axis_y >> m_axis_z
       >> m_axis_coord_sys
       >> m_pivot_x >> m_pivot_y >> m_pivot_z
       >> m_pivot_coord_sys;
}
