/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/math/fpclassify.hpp>
#include <Eigen/LU>

#include "SetOrientation.hh"
#include "../serialization/Eigen.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


SetOrientation::SetOrientation(
    ObjectID target,
    const Eigen::Vector3f &orientation,
    CoordinateSystem coord_sys) throw(std::logic_error)
    : m_target(target),
      m_orientation(orientation),
      m_coord_sys(coord_sys)
{
    if( m_orientation.isApprox(Eigen::Vector3f(0,0,0)) )
        throw std::logic_error("Orientation must be a non-zero vector");

    m_orientation.normalize();
}


SetOrientation::SetOrientation() throw()
{
}


SetOrientation::SetOrientation(const SetOrientation &action) throw()
    : m_target(action.m_target),
      m_orientation(action.m_orientation),
      m_coord_sys(action.m_coord_sys)
{
}


SetOrientation::~SetOrientation() throw()
{
}


Action *SetOrientation::clone() const
{
    return new SetOrientation(*this);
}


void SetOrientation::execute(
    ServerExecutionContext *context)
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Failed to set orientation, target object not found");

    // Check for infinity and NaN
    for( int i = 0; i < 3; i++ )
        if( !boost::math::isfinite(m_orientation(i)) )
            throw std::logic_error(
                "Failed to set orientation: orientation "
                "vector cannot contain infinity or NaN");

    Eigen::Vector3f orient;
    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
            orient = ptr->get_transformation().linear() * m_orientation;
            break;

        case PARENT_COORDINATES:
            orient = m_orientation;
            break;

        case WORLD_COORDINATES:
            orient = Eigen::Transform3f(
                ptr->get_parent_mtow().inverse(Eigen::Isometry)).linear() *
                m_orientation;
            break;

        default:
            throw std::runtime_error(
                "Failed to set orientation, unsupported coordinate system");
            break;
    }

    orient.normalize(); // Important! The equality test will fail otherwise
    Eigen::Transform3f m = ptr->get_transformation();
    Eigen::Vector3f old_orient(m(0,0), m(1,0), m(2,0));
    float x = m(0,3), y = m(1,3), z = m(2,3);

    if( old_orient.isApprox(orient, 0.001) )
    {
        // It's already correctly oriented
        return;
    }
    else if( old_orient.isApprox(-orient, 0.001) )
    {
        // The current and requested orientation are parallel but with opposite
        // directions; in this case the there is not a unique solution. The
        // requested orientation can be achieved by rotating +-180 degrees
        // about either local Z or local Y. By convention we will choose the
        // rotate-around-local-Z solution.
        ptr->set_transformation(
            Eigen::AngleAxisf(M_PI, m.linear().col(2)) * m);
    }
    else
    {
        Eigen::Vector3f axis = orient.cross(old_orient);
        axis.normalize();
        float theta = acosf(
            old_orient.dot(orient) / (old_orient.norm()*orient.norm()) );

        ptr->set_transformation(
            Eigen::Translation3f(x, y, z) *
            Eigen::AngleAxisf(-theta, axis) *
            Eigen::Translation3f(-x, -y, -z) *
            m);
    }
#endif
}


void SetOrientation::save(SerializationInterface &ar) const
{
    ar << m_target << m_orientation << m_coord_sys;
}

void SetOrientation::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_orientation >> m_coord_sys;
}
