/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/math/fpclassify.hpp>

#include "SetPosition.hh"
#include "serialization/Eigen.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


SetPosition::SetPosition(
    ObjectID target,
    const Eigen::Vector3f &pos,
    CoordinateSystem coord_sys) throw()
    : m_target(target),
      m_pos(pos),
      m_coord_sys(coord_sys)
{
}


SetPosition::SetPosition() throw()
{

}


SetPosition::SetPosition(const SetPosition &action) throw()
    : m_target(action.m_target),
      m_pos(action.m_pos),
      m_coord_sys(action.m_coord_sys)
{
}


SetPosition::~SetPosition() throw()
{
}


Action *SetPosition::clone() const
{
    return new SetPosition(*this);
}


void SetPosition::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Failed to set position, target object not found");

    // Check for infinity and NaN
    for( int i = 0; i < 3; i++ )
        if( !boost::math::isfinite(m_pos(i)) )
            throw std::logic_error(
                "Failed to set position: position "
                "cannot contain infinity or NaN");

    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
        {
            Eigen::Transform3f tmp;
            tmp = Eigen::Translation3f(m_pos(0), m_pos(1), m_pos(2));
            ptr->apply_transformation(tmp);
            break;
        }

        case PARENT_COORDINATES:
            ptr->set_position(m_pos);
            break;

        case WORLD_COORDINATES:
            ptr->set_world_position(m_pos);
            break;

        default:
            throw std::runtime_error(
                "Failed to set position, unsupported coordinate system");
            break;
    }
#endif
}


void SetPosition::save(SerializationInterface &ar) const
{
    ar << m_target << m_pos << m_coord_sys;
}

void SetPosition::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_pos >> m_coord_sys;
}
