/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/math/fpclassify.hpp>
#include <stdexcept>

#include "SetTransformation.hh"
#include "../serialization/Eigen.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;



SetTransformation::SetTransformation() throw()
{
}


SetTransformation::SetTransformation(
    ObjectID target,
    const Eigen::Transform3f &m,
    CoordinateSystem coord_sys) throw()
    : m_target(target),
      m_transformation(m),
      m_coord_sys(coord_sys)
{
}


Action *SetTransformation::clone() const
{
    return new SetTransformation(m_target, m_transformation, m_coord_sys);
}


void SetTransformation::execute(
    ServerExecutionContext *context)
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Failed to set transformation, target object not found");

    // Check for infinity and NaN
    for( int r = 0; r < 4; ++r )
        for( int c = 0; c < 4; ++c )
            if( !boost::math::isfinite(m_transformation(r,c)) )
                throw std::logic_error(
                    "Failed to set transformation: the transformation "
                    "matrix cannot contain infinity or NaN");

    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
            ptr->apply_transformation(m_transformation);
            break;

        case PARENT_COORDINATES:
            ptr->set_transformation(m_transformation);
            break;

        case WORLD_COORDINATES:
            ptr->set_transformation(
                Eigen::Transform3f(
                    ptr->get_parent_mtow().inverse(Eigen::Isometry)) *
                m_transformation);
            break;

        default:
            throw std::runtime_error(
                "Failed to set transformation: unsupported coordinate system");
            break;
    }
#endif
}


void SetTransformation::save(SerializationInterface &ar) const
{
    ar << m_target << m_transformation << m_coord_sys;
}

void SetTransformation::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_transformation >> m_coord_sys;
}
