/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <fstream>
#include <cstring>

#include "UploadFile.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#endif


using namespace peekabot;


UploadFile::UploadFile(const std::string &path)
    : m_path(path),
      m_file_len(0)
{
    std::ifstream ifs(path.c_str(), std::ios::binary);

    if( !ifs )
        throw std::runtime_error(
            "Failed to open file '" + path + "' for reading");

    ifs.seekg(0, std::ios::end);
    m_file_len = (boost::uint64_t)ifs.tellg();
    ifs.seekg(0, std::ios::beg);

    if( m_file_len > 0 )
    {
        m_file_data.reset(new boost::uint8_t[m_file_len]);
        ifs.read((char *)m_file_data.get(), m_file_len);
    }
}


UploadFile::UploadFile()
    : m_file_len(0)
{
}


UploadFile::UploadFile(const UploadFile &other)
    : m_path(other.m_path),
      m_file_len(other.m_file_len)
{
    if( m_file_len > 0 )
    {
        m_file_data.reset(new boost::uint8_t[m_file_len]);
        std::memcpy(m_file_data.get(), other.m_file_data.get(), m_file_len);
    }
}


UploadFile::~UploadFile() throw()
{
}


Action *UploadFile::clone() const
{
    return new UploadFile(*this);
}


void UploadFile::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    context->add_remote_file(
        m_path, m_file_data.get(), m_file_len);
#endif
}


void UploadFile::save(SerializationInterface &ar) const
{
    ar << m_path << m_file_len;

    if( m_file_len > 0 )
        ar.save_binary(m_file_data.get(), m_file_len);
}


void UploadFile::load(DeserializationInterface &ar)
{
    ar >> m_path >> m_file_len;

    if( m_file_len > 0 )
    {
        m_file_data.reset(new boost::uint8_t[m_file_len]);
        ar.load_binary(m_file_data.get(), m_file_len);
    }
}
