/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <iostream>

#include "ClientImpl.hh"
#include "ServerConnection.hh"
#include "ClientExecutionContext.hh"
#include "../IDAllocator.hh"
#include "../Init.hh"
#include "../actions/ActionMonitor.hh"
#include "../actions/Bundle.hh"
#include "../actions/NoOp.hh"
#include "ActionRecorder.hh"
#include "Exceptions.hh"
#include "Status.hh"


using namespace peekabot;
using namespace peekabot::client;


ClientImpl::ClientImpl() throw()
    : m_master(0),
      m_recorder(0),
      m_request_id_allocator(new DefaultIDAllocator<uint32_t>())
{
    init();
}

ClientImpl::~ClientImpl() throw()
{
    disconnect_all();
}


void ClientImpl::connect_master(const std::string &hostname, unsigned int port)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_master != 0 )
        throw AlreadyConnected(
            "The client is already connected to a master server");

    ServerConnection *conn = new ServerConnection(shared_from_this());
    try
    {
        conn->connect(hostname, port, false);
        m_master = conn;
    }
    catch(...)
    {
        delete conn;
        throw;
    }
}


void ClientImpl::disconnect_master()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_master == 0 )
        throw std::runtime_error("Master not connected");
    else
    {
        delete m_master;
        m_master = 0;
    }
}


void ClientImpl::disconnect_all()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( is_master_connected() )
        disconnect_master();

    if( is_recording() )
        stop_recording();
}


bool ClientImpl::is_master_connected() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    return m_master != 0;
}


void ClientImpl::start_recording(const std::string &filename)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_recorder != 0 )
        throw std::runtime_error("Already recording");
    else
        m_recorder = new ActionRecorder(shared_from_this(), filename);
}


void ClientImpl::stop_recording()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_recorder == 0 )
        throw std::runtime_error("Not recording");
    else
    {
        delete m_recorder;
        m_recorder = 0;
    }
}


bool ClientImpl::is_recording() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    return m_recorder != 0;
}


void ClientImpl::flush_recorder()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_recorder )
        m_recorder->flush();
}



uint32_t ClientImpl::allocate_request_id() throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    return m_request_id_allocator->allocate();
}


void ClientImpl::release_request_id(uint32_t request_id) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    m_request_id_allocator->release(request_id);
}


void ClientImpl::execute_action(
    boost::shared_ptr<Action> action) throw()
{
    ClientExecutionContext context(this);

    try
    {
        action->execute(&context);
    }
    catch(std::exception &e)
    {
        std::cerr << "WARNING: peekabot client caught an exception "
                  << "when processing data from the server.\n"
                  << "  what(): " << e.what()<< std::endl;
    }
    catch(...)
    {
        std::cerr << "WARNING: peekabot client caught unexpected exception "
                  << "when processing data from the server" << std::endl;
    }
}


//
//
//

void ClientImpl::dispatch_action(
    boost::shared_ptr<Action> action,
    Status *s,
    bool bypass_bundling,
    bool master_only)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    BundleData *bd = m_bundle_data.get();

    if( s )
    {
        uint32_t request_id = allocate_request_id();
        boost::shared_ptr<OperationStatus> op_status =
            register_status_request(request_id);

        *s = op_status;

        if( m_master )
        {
            boost::shared_ptr<Action> mon_action(
                new ActionMonitor(action, request_id));

            if( bd && !bypass_bundling )
                bd->m_master_bundle->add_action(mon_action);
            else
                m_master->dispatch_action(mon_action);
        }
        else
        {
            // It's a monitor action, but there were no transports around to
            // handle it - fail it (rather than letting it be pending forever)
            report_action_status(
                request_id,
                OPERATION_FAILED,
                "Master not connected");
        }
    }
    else
    {
        if( bd && !bypass_bundling )
            bd->m_master_bundle->add_action(action);
        else if( m_master )
            m_master->dispatch_action(action);
    }


    if( !master_only )
    {
        if( bd && !bypass_bundling )
        {
            bd->m_slave_bundle->add_action(action);
        }
        else
        {
            if( m_recorder )
            {
                m_recorder->dispatch_action(action);
            }

            /*for( each slave s )
              s->dispatch_action(action);*/
        }
    }
}


void ClientImpl::dispatch_action(
    Action *action,
    Status *s,
    bool bypass_bundling,
    bool master_only)
{
    dispatch_action(
        boost::shared_ptr<Action>(action), s,
        bypass_bundling, master_only);
}


boost::shared_ptr<OperationResult> ClientImpl::dispatch_get_action(
    boost::shared_ptr<Action> action,
    uint32_t request_id,
    bool bypass_bundling)
{
    // NOTE: that we only dispatch get actions to the master

    boost::recursive_mutex::scoped_lock lock(m_mutex);

    boost::shared_ptr<OperationResult> ret =
        register_result_request(request_id);
    BundleData *bd = m_bundle_data.get();

    if( m_master )
    {
        boost::shared_ptr<Action> mon_action(
            new ActionMonitor(action, request_id));

        if( bd )
            bd->m_master_bundle->add_action(mon_action);
        else
            m_master->dispatch_action(mon_action);
    }
    else
    {
        // There's no master connected to handle the get request - fail the
        // action, rather than letting it be pending forever
        report_action_status(
            request_id,
            OPERATION_FAILED,
            "Master not connected");
    }

    return ret;
}


boost::shared_ptr<OperationResult> ClientImpl::dispatch_get_action(
    Action *action,
    uint32_t request_id,
    bool bypass_bundling)
{
    return dispatch_get_action(
        boost::shared_ptr<Action>(action), request_id, bypass_bundling);
}


void ClientImpl::begin_bundle()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_bundle_data.get() != 0 )
        throw std::runtime_error(
            "begin_bundle() called with an already active bundle");

    m_bundle_data.reset(new BundleData);
    m_bundle_data.get()->m_master_bundle.reset(new Bundle);
    m_bundle_data.get()->m_slave_bundle.reset(new Bundle);
}


boost::shared_ptr<Action> ClientImpl::end_bundle()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_bundle_data.get() == 0 )
        throw std::runtime_error(
            "end_bundle() called without an active bundle");

    if( m_recorder )
        m_recorder->dispatch_action(m_bundle_data.get()->m_slave_bundle);

    /*for( each slave s )
      s->dispatch_action(action);*/

    boost::shared_ptr<Action> tmp(m_bundle_data.get()->m_master_bundle);
    m_bundle_data.reset();

    // IMPORTANT! It is vital that the third master_only argument is set to
    // true here, otherwise a get an internal loop and end up duplicating all
    // the data.
    return tmp;
}


bool ClientImpl::is_bundling() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_bundle_data.get() != 0;
}


void ClientImpl::sync_master()
{
    // NOTE: Watch out for deadlocks here! The status is reported by a
    // different thread, so we can hold a lock the mutex while waiting for the
    // operation to finish.

    Status s;

    {
        boost::recursive_mutex::scoped_lock lock(m_mutex);

        if( m_master )
        {
            uint32_t request_id = allocate_request_id();
            boost::shared_ptr<OperationStatus> op_status =
                register_status_request(request_id);

            s = op_status;

            boost::shared_ptr<Action> mon_action(
                new ActionMonitor(
                    boost::shared_ptr<Action>(new NoOp()), request_id));

            m_master->dispatch_action(mon_action);
        }
        else
            return;
    }

    s.wait_until_completed();
}


boost::shared_ptr<OperationStatus>
ClientImpl::register_status_request(uint32_t request_id) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    // Note: if the connection is torn down during this call, the
    // "shutdown sequence" will wait for the requests mutex and
    // then alert the Status object we were disconnected AFTER this
    // method invokation has ended.

    boost::shared_ptr<OperationStatus> ptr(new OperationStatus());
    m_requests.insert(std::make_pair(request_id, ptr));

    return ptr;
}


boost::shared_ptr<OperationResult>
ClientImpl::register_result_request(uint32_t request_id) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    // Note: if the connection is torn down during this call, the
    // "shutdown sequence" will wait for the requests mutex and
    // then alert the Status object we were disconnected AFTER this
    // method invokation has ended.

    boost::shared_ptr<OperationResult> ptr(new OperationResult());
    m_requests.insert(std::make_pair(request_id, ptr));

    return ptr;
}


void ClientImpl::flush_master() throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_master )
        m_master->flush();
}


void ClientImpl::flush_all() throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_master )
        flush_master();

    if( m_recorder )
        flush_recorder();
}



void ClientImpl::report_action_status(
    boost::uint32_t request_id,
    OperationOutcome outcome,
    const std::string &error_msg) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    RequestMap::iterator it = m_requests.find(request_id);

    if( it != m_requests.end() )
    {
        it->second->set_outcome(outcome, error_msg);

        m_requests.erase(it);

        m_request_id_allocator->release(request_id);
    }
    // TODO: else log error
}


void ClientImpl::report_action_result(
    boost::uint32_t request_id,
    const Any &result) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    RequestMap::iterator it = m_requests.find(request_id);

    if( it != m_requests.end() )
    {
        boost::shared_ptr<OperationResult> r =
            boost::dynamic_pointer_cast<OperationResult>(it->second);

        if( !r.get() )
            return; // TODO: log error

        r->set_result(result);
    }
    // TODO: else log error
}


void ClientImpl::report_disconnected_transport(Transport *t)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    // If the disconnected transport was the "master",
    // wake all threads waiting on statuses, and let them
    // know we disconnected
    if( t == m_master )
    {
        for( ClientImpl::RequestMap::iterator it = m_requests.begin();
             it != m_requests.end(); ++it )
        {
            it->second->client_disconnected();
        }

        delete m_master;
        m_master = 0;
    }
    else if( t == m_recorder )
    {
        delete m_recorder;
        m_recorder = 0;
    }
}
