/*
 * Copyright Staffan Gimåker 2006-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_CLIENT_CLIENT_IMPL_HH_INCLUDED
#define PEEKABOT_CLIENT_CLIENT_IMPL_HH_INCLUDED


#include <map>
#include <string>
#include <boost/cstdint.hpp>
#include <boost/thread.hpp>
#include <boost/utility.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>

#include "../Any.hh"
#include "../Types.hh"
#include "Status.hh"
#include "Result.hh"


namespace peekabot
{

    template<class> class IDAllocator;
    class Action;
    class Bundle;

    namespace client
    {
        class Transport;
        class ClientExecutionContext;
        class ObjectProxyBase;
        class DelayedDispatch;



        /**
         * \internal
         *
         * \brief Provides a middle-level interface, seated below \c Proxy,
         * for communicating with a peekabot server. Deals with the intrinsic
         * data pertaining to status and result requests and talks to lower
         * level interfaces.
         *
         * Acts a middle man between \c Proxy, whose responsibility is
         * providing a public interface generating the actions to be sent,
         * and the classes responsible for actually serializing and sending
         * the data.
         *
         * As such, it is the responsible party for handling and executing
         * incoming actions.
         */
        class ClientImpl : public boost::noncopyable,
                           public boost::enable_shared_from_this<ClientImpl>
        {
            friend class Transport;
            friend class ClientExecutionContext;
            friend class ObjectProxyBase;

        public:
            ClientImpl() throw();

            virtual ~ClientImpl() throw();

            /**
             * \brief Connect the proxy to a peekabot server.
             *
             * Unlike most other proxy methods, this particular method
             * is \e synchronous. It will initiate a connection with the server
             * given, and go through the authentication process before returning.
             *
             * If the proxy is already connected, the method will fail and the
             * already established connection will remain unaffected.
             *
             * Commands (such as \c set_transformation()) may be called \e before
             * connecting to a server. In that case, those commands will be put
             * on hold and sent when a connection is established.
             *
             * \param hostname The hostname or IP address of the machine running
             * the peekabot server.
             * \param port The port on which the server is running on that
             * machine.
             */
            void connect_master(const std::string &hostname, unsigned int port);

            /**
             * \remarks This operation might \e block for some time before
             * returning.
             * \remarks If an active transaction exists, it will be discarded.
             */
            void disconnect_master();

            bool is_master_connected() const;

            void flush_master() throw();

            /**
             * \brief Block until all outbound actions have been sent.
             *
             * If the connection is terminated during a call to flush, the
             * method will return, even though not all actions were sent.
             */
            void flush_all() throw();

            void dispatch_action(
                boost::shared_ptr<Action> action,
                Status *s,
                bool bypass_bundling,
                bool master_only);

            void dispatch_action(
                Action *action,
                Status *s,
                bool bypass_bundling,
                bool master_only);

            boost::shared_ptr<OperationResult> dispatch_get_action(
                boost::shared_ptr<Action> action,
                uint32_t request_id,
                bool bypass_bundling);

            boost::shared_ptr<OperationResult> dispatch_get_action(
                Action *action,
                uint32_t request_id,
                bool bypass_bundling);

            void begin_bundle();

            boost::shared_ptr<Action> end_bundle();

            bool is_bundling() const;

            void sync_master();


            void disconnect_all();


            void start_recording(const std::string &filename);

            void stop_recording();

            bool is_recording() const;

            void flush_recorder();


            uint32_t allocate_request_id() throw();

            boost::shared_ptr<OperationResult>
            register_result_request(uint32_t request_id) throw();

        private:
            void execute_action(boost::shared_ptr<Action> action) throw();

            void release_request_id(uint32_t request_id) throw();

            void report_disconnected_transport(Transport *t);

            boost::shared_ptr<OperationStatus>
            register_status_request(uint32_t request_id) throw();



        private:
            mutable boost::recursive_mutex m_mutex;

            Transport *m_master;
            Transport *m_recorder;

            struct BundleData
            {
                boost::shared_ptr<Bundle> m_master_bundle;
                boost::shared_ptr<Bundle> m_slave_bundle;
            };

            boost::thread_specific_ptr<BundleData> m_bundle_data;


            //
            // --- Request-related stuff ---
            //
            typedef std::map<
                uint32_t, boost::shared_ptr<OperationStatus> > RequestMap;

            RequestMap m_requests;

            boost::scoped_ptr<IDAllocator<uint32_t> > m_request_id_allocator;

            void report_action_status(
                boost::uint32_t request_id,
                OperationOutcome outcome,
                const std::string &error_msg) throw();

            void report_action_result(
                boost::uint32_t request_id,
                const Any &result) throw();
        };

    }
}


#endif // PEEKABOT_CLIENT_CLIENT_IMPL_HH_INCLUDED
