/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_CLIENT_RECORDING_HH_INCLUDED
#define PEEKABOT_CLIENT_RECORDING_HH_INCLUDED


#include <boost/shared_ptr.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

#include "../Visibility.hh"


namespace peekabot
{
    namespace client
    {
        class RecordingImpl;

        /**
         * \brief Class that controls playback of peekabot recordings.
         *
         * Upon creation, playback is paused.
         */
        class PEEKABOT_API Recording
        {
        public:
            Recording(boost::shared_ptr<RecordingImpl> impl);

            /**
             * \brief Start playback.
             *
             * If the recording is finished or already playing, this is a
             * no-op.
             */
            void play();

            /**
             * \brief Pause playback.
             */
            void pause();

            /**
             * \brief Returns true if playback is currently paused.
             */
            bool is_paused() const;

            /**
             * \brief Returns true if playback of the recording has finished.
             */
            bool is_finished() const;

            /**
             * \brief Step \a n operations.
             *
             * If playback isn't paused, this is a no-op.
             *
             * \pre <tt>n > 0</tt>
             *
             * \remarks Note that bundles count as one operation, thus in
             * combination with stepping they're a formidable tool.
             */
            void step(size_t n = 1);

            /**
             * \brief Step the specified amount of time.
             *
             * If playback isn't paused, this is a no-op.
             *
             * \pre \c td is positive.
             */
            void step(const boost::posix_time::time_duration &td);

            /**
             * \brief Change the playback speed.
             *
             * \pre <tt>factor > 0</tt>
             */
            void set_playback_speed(double factor);

            /**
             * \brief Get the current playback speed.
             */
            double get_playback_speed() const;

            /**
             * \brief Returns the elapsed playback time.
             */
            boost::posix_time::time_duration elapsed() const;

            /**
             * \brief Returns the recordings total duration.
             */
            boost::posix_time::time_duration duration() const;

        private:
            boost::shared_ptr<RecordingImpl> m_impl;
        };
    }

    using client::Recording;
}


#endif // PEEKABOT_CLIENT_RECORDING_HH_INCLUDED
