/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "GridProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/AddObject.hh"
#include "../../actions/Assign.hh"
#include "../../actions/SetProp.hh"


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ GridProxyBase implementation -------------------
//


GridProxyBase::GridProxyBase()
{
}


GridProxyBase::GridProxyBase(GridProxyBase &p)
    : ObjectProxyBase(p),
      ScalableProxyBase(p),
      LineBasedProxyBase(p)
{
}

//

DelayedDispatch GridProxyBase::set_segment_count(boost::uint32_t n)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), GRID_SEGMENTS_PROP, n));
}

DelayedDispatch GridProxyBase::set_grid_type(GridType type)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), GRID_TYPE_PROP, type));
}

DelayedDispatch GridProxyBase::set_central_angle(float central_angle)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), GRID_CENTRAL_ANGLE_PROP, central_angle));
}


//
// ------------------ GridProxy implementation -------------------
//


GridProxy::GridProxy()
{
}


GridProxy::GridProxy(GridProxyBase &p)
    : ObjectProxyBase(p),
      GridProxyBase(p)
{
}


GridProxy &GridProxy::operator=(const GridProxy &p)
{
    return *this = (GridProxyBase &)p;
}


GridProxy &GridProxy::operator=(const GridProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}


DelayedDispatch GridProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(get_object_id(p)),
                   get_object_id(), GRID_OBJECT));
}


DelayedDispatch GridProxy::assign(
    PeekabotClient &client, const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), GRID_OBJECT));
}


DelayedDispatch GridProxy::assign(
    const ObjectProxyBase &parent, 
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(), new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), GRID_OBJECT));
}


DelayedDispatch GridProxy::add(
    PeekabotClient &client,
    const std::string &path,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(path), conflict_policy,
                      get_object_id(), GRID_OBJECT));
}


DelayedDispatch GridProxy::add(
    const ObjectProxyBase &parent, 
    const std::string &name,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(get_object_id(parent), name),
                      conflict_policy, get_object_id(), GRID_OBJECT));
}
