/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/thread/mutex.hpp>
#include <Eigen/Core>

#include "ObjectProxy.hh"
#include "../PeekabotClient.hh"
#include "../../IDAllocator.hh"
#include "../../PropKeys.hh"

#include "../../actions/RegisterPseudonym.hh"
#include "../../actions/DeregisterPseudonym.hh"
#include "../../actions/SetTransformation.hh"
#include "../../actions/SetPosition.hh"
#include "../../actions/SetOrientation.hh"
#include "../../actions/SetPose.hh"
#include "../../actions/RemoveObject.hh"
#include "../../actions/ClearChildren.hh"
#include "../../actions/LoadScene.hh"
#include "../../actions/GetTransformation.hh"
#include "../../actions/GetPosition.hh"
#include "../../actions/GetOrientation.hh"
#include "../../actions/GetChildren.hh"
#include "../../actions/SetRotation.hh"
#include "../../actions/RearrangeObject.hh"
#include "../../actions/Rotate.hh"
#include "../../actions/Translate.hh"
#include "../../actions/SetProp.hh"


using namespace peekabot;
using namespace peekabot::client;


namespace
{
    /*
     * A simple locking-decorator for another IDAllocator class, making it
     * thread-safe.
     */
    template<
        class IDType,
        template<class> class IDAllocatorImpl = DefaultIDAllocator>
    class ThreadSafeIDAllocator : public IDAllocator<IDType>
    {
    public:
        virtual IDType allocate()
        {
            boost::mutex::scoped_lock lock(m_mutex);
            return m_backend.allocate();
        }

        virtual void release(IDType id)
        {
            boost::mutex::scoped_lock lock(m_mutex);
            m_backend.release(id);
        }

    private:
        IDAllocatorImpl<IDType> m_backend;
        boost::mutex m_mutex;
    };

    ThreadSafeIDAllocator<ObjectID> pseudonym_allocator;
}

//
// ------------------- ObjectProxyBase implementation ---------------------
//

ObjectProxyBase::ObjectProxyBase()
{
}


ObjectProxyBase::ObjectProxyBase(const ObjectProxyBase &p)
    : PeekabotProxyBase(p),
      m_pseudonym(p.get_pseudonym())
{
}


ObjectProxyBase::~ObjectProxyBase()
{
    // Reset pseudonym (will release it if no one else is using it)
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( m_pseudonym )
    {
        // Pseudonym previously assigned, release it?
        if( m_pseudonym.unique() )
        {
            pseudonym_allocator.release(*m_pseudonym);
            dispatch_action(new DeregisterPseudonym(*m_pseudonym), 0);
        }

        m_pseudonym.reset();
    }
}


bool ObjectProxyBase::operator==(const ObjectProxyBase &p) const
{
    return unchecked_get_client_impl() == p.unchecked_get_client_impl() &&
        get_object_id() == p.get_object_id();
}


bool ObjectProxyBase::operator!=(const ObjectProxyBase &p) const
{
    return !(p == *this);
}


ObjectID ObjectProxyBase::get_object_id() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    if( !is_assigned() )
        throw std::logic_error("The object proxy is unassigned, "
                               "i.e. not yet tied to a remote object");
    assert( m_pseudonym );
    return *m_pseudonym;
}


ObjectID ObjectProxyBase::get_object_id(const ObjectProxyBase &p)
{
    return p.get_object_id();
}


void ObjectProxyBase::unchecked_assign(
    boost::shared_ptr<ClientImpl> client, boost::shared_ptr<ObjectID> id)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    // NOTE: Order matters here - the pseudonym must be released before we
    // change the client (since it might be a different client!)
    if( m_pseudonym )
    {
        assert( client );

        // Pseudonym previously assigned, release it?
        boost::shared_ptr<ObjectID> tmp = m_pseudonym;
        m_pseudonym.reset();

        if( tmp.unique() )
        {
            pseudonym_allocator.release(*tmp);
            dispatch_action(new DeregisterPseudonym(*tmp), 0);
        }
    }

    set_client_impl(client);
    m_pseudonym = id;
}


void ObjectProxyBase::unchecked_assign(const ObjectProxyBase &other)
{
    boost::shared_ptr<ClientImpl> client;
    boost::shared_ptr<ObjectID> pid;

    {
        boost::recursive_mutex::scoped_lock lock(other.m_mutex);
        client = other.unchecked_get_client_impl();
        pid = other.get_pseudonym();
    }

    unchecked_assign(client, pid);
}


boost::shared_ptr<ObjectID> ObjectProxyBase::get_pseudonym() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_pseudonym;
}


boost::shared_ptr<ObjectID> ObjectProxyBase::get_pseudonym(
    const ObjectProxyBase &p)
{
    return p.get_pseudonym();
}


boost::shared_ptr<ObjectID> ObjectProxyBase::allocate_pseudonym()
{
    boost::shared_ptr<ObjectID> ret(new ObjectID);
    *ret = pseudonym_allocator.allocate();
    return ret;
}


//


DelayedDispatch ObjectProxyBase::set_transformation(
    float x1, float y1, float z1, float p1,
    float x2, float y2, float z2, float p2,
    float x3, float y3, float z3, float p3,
    CoordinateSystem coord_sys)
{
    Eigen::Transform3f M;
    M(0,0) = x1; M(0,1) = y1; M(0,2) = z1; M(0,3) = p1;
    M(1,0) = x2; M(1,1) = y2; M(1,2) = z2; M(1,3) = p2;
    M(2,0) = x3; M(2,1) = y3; M(2,2) = z3; M(2,3) = p3;
    M(3,0) =  0; M(3,1) =  0; M(3,2) =  0; M(3,3) =  1;

    return DelayedDispatch(
        get_client_impl(),
        new SetTransformation(get_object_id(), M, coord_sys));
}


DelayedDispatch ObjectProxyBase::set_transformation(
    const float *m, bool row_major,
    CoordinateSystem coord_sys)
{
    const std::size_t i_stride = (row_major ? 4 : 1);
    const std::size_t j_stride = (row_major ? 1 : 4);

    Eigen::Transform3f M;
    M.matrix().row(3) = Eigen::Vector4f(0,0,0,1);
    for( std::size_t i = 0; i < 3; ++i )
        for( std::size_t j = 0; j < 4; ++j )
            M(i,j) = m[i*i_stride + j*j_stride];

    return DelayedDispatch(
        get_client_impl(),
        new SetTransformation(get_object_id(), M, coord_sys));
}


DelayedDispatch ObjectProxyBase::set_position(
    float x, float y, float z,
    CoordinateSystem coord_sys)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetPosition(
            get_object_id(),
            Eigen::Vector3f(x,y,z),
            coord_sys));
}


DelayedDispatch ObjectProxyBase::translate(
    float x, float y, float z,
    CoordinateSystem coord_sys)
{
    return DelayedDispatch(
        get_client_impl(),
        new Translate(get_object_id(), x, y, z, coord_sys));
}


DelayedDispatch ObjectProxyBase::set_orientation(
    float vx, float vy, float vz,
    CoordinateSystem coord_sys)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetOrientation(
            get_object_id(),
            Eigen::Vector3f(vx,vy,vz),
            coord_sys));
}


DelayedDispatch ObjectProxyBase::set_rotation(
    float yaw, float pitch, float roll,
    CoordinateSystem coord_sys)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetRotation(get_object_id(), yaw, pitch, roll, coord_sys));
}


DelayedDispatch ObjectProxyBase::rotate(
    float rad,
    float axis_x, float axis_y, float axis_z,
    float pivot_x, float pivot_y, float pivot_z,
    CoordinateSystem axis_coord_sys,
    CoordinateSystem pivot_coord_sys)
{
    if( axis_x*axis_x + axis_y*axis_y + axis_z*axis_z == 0 )
        throw std::logic_error("Rotational axis must be non-zero");

    return DelayedDispatch(
        get_client_impl(),
        new Rotate(
            get_object_id(), rad,
            axis_x, axis_y, axis_z, axis_coord_sys,
            pivot_x, pivot_y, pivot_z, pivot_coord_sys));
}


DelayedDispatch ObjectProxyBase::set_pose(
    float x, float y, float z,
    float yaw, float pitch, float roll,
    CoordinateSystem coord_sys)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetPose(
            get_object_id(),
            x, y, z,
            yaw, pitch, roll, coord_sys));
}


DelayedDispatch ObjectProxyBase::set_opacity(float opacity, bool absolute)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), OPACITY_PROP, opacity));
}


DelayedDispatch ObjectProxyBase::set_visibility(bool visible)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), HIDDEN_PROP, !visible));
}


DelayedDispatch ObjectProxyBase::set_layer(
    unsigned int layer, bool recursive)
{
    if( layer < 1 || layer > NUMBER_OF_LAYERS )
        throw std::range_error("Invalid layer number");

    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), LAYER_PROP,
                    boost::uint8_t(layer-1), recursive));
}


DelayedDispatch ObjectProxyBase::set_color(
    float r, float g, float b, bool recursive)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), COLOR_PROP,
                    RGBColor(r,g,b), recursive));
}


DelayedDispatch ObjectProxyBase::set_name(const std::string &name)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), NAME_PROP, name));
}


DelayedDispatch ObjectProxyBase::remove()
{
    return DelayedDispatch(
        get_client_impl(),
        new RemoveObject(get_object_id()));
}


DelayedDispatch ObjectProxyBase::clear()
{
    return DelayedDispatch(
        get_client_impl(),
        new ClearChildren(get_object_id()));
}


DelayedDispatch ObjectProxyBase::rearrange(
    const ObjectProxyBase &new_parent,
    bool retain_world_pose,
    NameConflictPolicy conflict_policy) const
{
    return DelayedDispatch(
        get_client_impl(),
        new RearrangeObject(
            PathIdentifier(get_object_id()),
            PathIdentifier(get_object_id(new_parent)),
            retain_world_pose, conflict_policy));
}


DelayedDispatch ObjectProxyBase::load_scene(
    const std::string &filename,
    NameConflictPolicy conflict_policy)
{
    return DelayedDispatch(
        get_client_impl(),
        new LoadScene(get_object_id(), filename, conflict_policy));
}


Result<Transformation> ObjectProxyBase::get_transformation(
    CoordinateSystem coord_sys) const
{
    uint32_t request_id = allocate_request_id();

    return Result<Transformation>(
        dispatch_get_action(
            new GetTransformation(
                request_id, get_object_id(), coord_sys), request_id) );
}


Result<Vector3> ObjectProxyBase::get_position(
    CoordinateSystem coord_sys) const
{
    uint32_t request_id = allocate_request_id();

    return Result<Vector3>(
        dispatch_get_action(
            new GetPosition(
                request_id, get_object_id(), coord_sys), request_id) );
}


Result<Vector3> ObjectProxyBase::get_orientation(
    CoordinateSystem coord_sys) const
{
    uint32_t request_id = allocate_request_id();

    return Result<Vector3>(
        dispatch_get_action(
            new GetOrientation(
                request_id, get_object_id(), coord_sys), request_id) );
}


Result<std::vector<std::string> > ObjectProxyBase::get_children() const
{
    uint32_t request_id = allocate_request_id();

    return Result<std::vector<std::string> >(
        dispatch_get_action(
            new GetChildren(request_id, get_object_id()), request_id) );
}


//
// ------------------- ObjectProxy implementation ---------------------
//


ObjectProxy::ObjectProxy()
{
}

ObjectProxy::ObjectProxy(const ObjectProxyBase &p)
    : ObjectProxyBase(p)
{
}


ObjectProxy &ObjectProxy::operator=(const ObjectProxy &p)
{
    return *this = (ObjectProxyBase &)p;
}


ObjectProxy &ObjectProxy::operator=(const ObjectProxyBase &p)
{
    unchecked_assign(p);
    return *this;
}

DelayedDispatch ObjectProxy::assign(
    PeekabotClient &client, const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    // We don't need any type checking for ObjectProxy base since it's the base
    // class for all concrete objects, so just registering a pseudonym will do
    return DelayedDispatch(
        get_client_impl(),
        new RegisterPseudonym(path, get_object_id()));
}

DelayedDispatch ObjectProxy::assign(
    const ObjectProxyBase &parent,
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    // We don't need any type checking for ObjectProxy base since it's the base
    // class for all concrete objects, so just registering a pseudonym will do
    return DelayedDispatch(
        get_client_impl(),
        new RegisterPseudonym(
            get_object_id(parent), rel_path, get_object_id()));
}
