/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "OccupancyGrid2DProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/AddObject.hh"
#include "../../actions/Assign.hh"
#include "../../actions/SetOccupancyGrid2DCells.hh"
#include "../../actions/SetProp.hh"
#include "../../actions/MiniBundle.hh"

#include <vector>
#include <Eigen/Core>


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ OccupancySet2D implementation -------------------
//

struct OccupancySet2D::Impl
{
    inline void add(float x, float y, float belief)
    {
        m_cells.push_back(std::make_pair(Eigen::Vector2f(x, y), belief));
    }

    inline void clear()
    {
        m_cells.clear();
    }

    inline std::size_t size() const
    {
        return m_cells.size();
    }

    inline bool empty() const
    {
        return m_cells.empty();
    }

    std::vector<std::pair<Eigen::Vector2f, float> > m_cells;
};


OccupancySet2D::OccupancySet2D()
    : m_impl(new OccupancySet2D::Impl)
{
}

OccupancySet2D::OccupancySet2D(const OccupancySet2D &other)
    : m_impl(new Impl(*other.m_impl))
{
}

OccupancySet2D::~OccupancySet2D()
{
}

OccupancySet2D &OccupancySet2D::operator=(const OccupancySet2D &other)
{
    m_impl.reset(new Impl(*other.m_impl));
    return *this;
}

void OccupancySet2D::add(float x, float y, float belief)
{
    m_impl->add(x, y, belief);
}

void OccupancySet2D::set_cell(float x, float y, float belief)
{
    add(x, y, belief);
}

void OccupancySet2D::clear()
{
    m_impl->clear();
}

std::size_t OccupancySet2D::size() const
{
    return m_impl->size();
}

bool OccupancySet2D::empty() const
{
    return m_impl->empty();
}


//
// ------------------ OccupancyGrid2DProxyBase implementation -------------------
//


OccupancyGrid2DProxyBase::OccupancyGrid2DProxyBase()
{
}


OccupancyGrid2DProxyBase::OccupancyGrid2DProxyBase(OccupancyGrid2DProxyBase &p)
    : ObjectProxyBase(p)
{
}

//

DelayedDispatch OccupancyGrid2DProxyBase::set_cells(
    const OccupancySet2D &cells)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetOccupancyGrid2DCells(get_object_id(), cells.m_impl->m_cells));
}


DelayedDispatch OccupancyGrid2DProxyBase::set_unoccupied_color(
    float r, float g, float b)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(
            get_object_id(), OG2D_UNOCCUPIED_COLOR_PROP, RGBColor(r,g,b)));
}


DelayedDispatch OccupancyGrid2DProxyBase::set_occupied_color(
    float r, float g, float b)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(
            get_object_id(), OG2D_OCCUPIED_COLOR_PROP, RGBColor(r,g,b)));
}


//
// ------------------ OccupancyGrid2DProxy implementation -------------------
//


OccupancyGrid2DProxy::OccupancyGrid2DProxy()
{
}


OccupancyGrid2DProxy::OccupancyGrid2DProxy(OccupancyGrid2DProxy &p)
    : OccupancyGrid2DProxyBase(p)
{
}


OccupancyGrid2DProxy &OccupancyGrid2DProxy::operator=(
    const OccupancyGrid2DProxy &p)
{
    return *this = (OccupancyGrid2DProxyBase &)p;
}


OccupancyGrid2DProxy &OccupancyGrid2DProxy::operator=(const
    OccupancyGrid2DProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}


DelayedDispatch OccupancyGrid2DProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(
            PathIdentifier(get_object_id(p)), get_object_id(), OG2D_OBJECT));
}


DelayedDispatch OccupancyGrid2DProxy::assign(
    PeekabotClient &client,
    const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), OG2D_OBJECT));
}


DelayedDispatch OccupancyGrid2DProxy::assign(
    const ObjectProxyBase &parent,
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(), new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), OG2D_OBJECT));
}


DelayedDispatch OccupancyGrid2DProxy::add(
    PeekabotClient &client,
    const std::string &path,
    float cell_size,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_size));

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(
            PathIdentifier(path), conflict_policy,
            get_object_id(), OG2D_OBJECT, args));
}


DelayedDispatch OccupancyGrid2DProxy::add(
    const ObjectProxyBase &parent,
    const std::string &name,
    float cell_size,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_size));

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(
            PathIdentifier(get_object_id(parent), name),
            conflict_policy, get_object_id(), OG2D_OBJECT, args));
}


DelayedDispatch OccupancyGrid2DProxy::add(
    PeekabotClient &client,
    const std::string &path,
    float cell_size,
    float unoccupied_r, float unoccupied_g, float unoccupied_b,
    float occupied_r, float occupied_g, float occupied_b,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_size));

    MiniBundle *b = new MiniBundle();
    b->add_action(
        new AddObject(
            PathIdentifier(path), conflict_policy,
            get_object_id(), OG2D_OBJECT, args));
    b->add_action(
        new SetProp(
            get_object_id(), OG2D_UNOCCUPIED_COLOR_PROP,
            RGBColor(unoccupied_r, unoccupied_g, unoccupied_b)));
    b->add_action(
        new SetProp(
            get_object_id(), OG2D_OCCUPIED_COLOR_PROP,
            RGBColor(occupied_r, occupied_g, occupied_b)));

    return DelayedDispatch(get_client_impl(), b);
}


DelayedDispatch OccupancyGrid2DProxy::add(
    const ObjectProxyBase &parent,
    const std::string &name,
    float cell_size,
    float unoccupied_r, float unoccupied_g, float unoccupied_b,
    float occupied_r, float occupied_g, float occupied_b,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_size));

    MiniBundle *b = new MiniBundle();
    b->add_action(
        new AddObject(
            PathIdentifier(get_object_id(parent), name),
            conflict_policy, get_object_id(), OG2D_OBJECT, args));
    b->add_action(
        new SetProp(
            get_object_id(), OG2D_UNOCCUPIED_COLOR_PROP,
            RGBColor(unoccupied_r, unoccupied_g, unoccupied_b)));
    b->add_action(
        new SetProp(
            get_object_id(), OG2D_OCCUPIED_COLOR_PROP,
            RGBColor(occupied_r, occupied_g, occupied_b)));

    return DelayedDispatch(get_client_impl(), b);
}
