/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "EnumPropMediator.hh"
#include "../SceneObject.hh"
#include "../props/EnumPropBase.hh"
#include "../Server.hh"
#include "../SceneTree.hh"

#include <cassert>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


EnumPropMediator::EnumPropMediator(
    PropInspectorController &pi, SceneObject *obj,
    EnumPropBase *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0),
      m_widget(0), m_lbl(0)
{
    std::vector<std::string> enum_values;
    m_prop->get_enum_values(enum_values);

    // Create and init widget
    post(
        boost::bind(&EnumPropMediator::create_widget, this,
                    m_obj->get_prop_name(m_prop_key),
                    m_prop->get_as_text(m_obj), enum_values,
                    m_prop->is_read_only()));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&EnumPropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void EnumPropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(
        boost::bind(&EnumPropMediator::destroy_widget, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&EnumPropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void EnumPropMediator::create_widget(
    std::string prop_name,
    std::string val, std::vector<std::string> enum_values,
    bool read_only)
{
    assert( !m_widget );
    assert( !m_lbl );

    m_widget = new Gtk::ComboBoxText();
    for( std::size_t i = 0; i < enum_values.size(); ++i )
        m_widget->append_text(enum_values[i]);
    m_widget->set_active_text(val);
    m_widget_set_conn = m_widget->signal_changed().connect(
        sigc::mem_fun(*this, &EnumPropMediator::on_widget_set));
    if( read_only )
        m_widget->set_sensitive(false);

    m_lbl = new Gtk::Label(prop_name + ":", 1.0, 0.5);

    add_prop_widgets(m_lbl, m_widget);
}


// Executed in the GUI thread
void EnumPropMediator::destroy_widget()
{
    assert( m_widget );
    assert( m_lbl );

    erase_prop_widgets(m_lbl, m_widget);
    delete m_widget;
    m_widget = 0;
    delete m_lbl;
    m_lbl = 0;

    delete this;
}


// Executed in the server thread
void EnumPropMediator::on_prop_set()
{
    post(
        boost::bind(
            &EnumPropMediator::update_widget, this,
            m_prop->get_as_text(m_obj)));
}


// Executed in the GUI thread
void EnumPropMediator::update_widget(std::string val)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    m_widget_set_conn.block();
    m_widget->set_active_text(val);
    m_widget_set_conn.unblock();
}


// Executed in the GUI thread
void EnumPropMediator::on_widget_set()
{
    ++m_n_updates_queued;

    server_post(
        boost::bind(
            &EnumPropMediator::set_prop_from_text, this, _1,
            m_object_id, m_prop_key, m_widget->get_active_text()));
}


void EnumPropMediator::set_prop_from_text(
    ServerData &sd, ObjectID object_id,
    PropKey prop_key, std::string val)
{
    SceneObject *obj = sd.m_scene->get_object(object_id);
    if( obj )
    {
        EnumPropBase *p = dynamic_cast<EnumPropBase *>(obj->get_prop(prop_key));
        if( p )
            p->set_from_text(val, m_obj);
    }
}
