/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ExplorerController.hh"
#include "MainWindowController.hh"
#include "Gui.hh"
#include "../ObjectTypes.hh"
#include "../Server.hh"
#include "../SceneTree.hh"
#include "../SceneObject.hh"

#include <cassert>
#include <boost/bind.hpp>
#include <boost/signals2.hpp>


using namespace peekabot;
using namespace peekabot::gui;


class ExplorerController::Cols : public Gtk::TreeModelColumnRecord
{
public:
    Cols()
    {
        add(m_name);
        add(m_icon);
        add(m_id);
    }

    Gtk::TreeModelColumn<std::string> m_name;
    Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > m_icon;
    Gtk::TreeModelColumn<ObjectID> m_id;
};


class ExplorerController::DndTreeStore : public Gtk::TreeStore
{
public:
    static Glib::RefPtr<DndTreeStore> create(Cols &cols)
    {
        return Glib::RefPtr<DndTreeStore>(new DndTreeStore(cols));
    }

    typedef boost::signals2::signal<void (
        ObjectID id, ObjectID new_parent_id)> RearrangeSignal;

    inline RearrangeSignal &rearrange_signal()
    {
        return m_rearrange_signal;
    }

private:
    DndTreeStore(Cols &cols) : Gtk::TreeStore(cols), m_cols(cols) {}

protected:
    virtual bool row_draggable_vfunc(const Gtk::TreeModel::Path &path) const
    {
        // Root nodes cannot be dragged
        return path.size() > 1;
    }

    virtual bool row_drop_possible_vfunc(
        const Gtk::TreeModel::Path &dest,
        const Gtk::SelectionData &selection_data) const
    {
        // Nodes cannot be dropped
        // 1) when dest is not a valid path (either a root node, or an empty
        //    path)
        // 2) on themselves
        // 3) on descendants of themselves
        // 4) on their parent

        assert( m_dragged_path.size() > 1 );
        Gtk::TreePath old_parent = m_dragged_path;
        old_parent.up();

        return ( dest.size() > 1 && // 1)
                 m_dragged_path != dest && // 2)
                 !m_dragged_path.is_ancestor(dest) && // 3)
                 old_parent != dest ); // 4)
    }

    virtual bool drag_data_get_vfunc(
        const Gtk::TreeModel::Path &path,
        Gtk::SelectionData &selection_data) const
    {
        m_dragged_path = path;
        return Gtk::TreeStore::drag_data_get_vfunc(path, selection_data);
    }

    virtual bool drag_data_received_vfunc(
        const Gtk::TreeModel::Path &dest,
        const Gtk::SelectionData &selection_data)
    {
        assert( dest.size() > 1 );

        Gtk::TreePath new_parent_tp = dest;
        new_parent_tp.up();
        Gtk::TreeModel::iterator it = get_iter(new_parent_tp);

        if( it )
            m_new_parent_id = (*it)[m_cols.m_id];

        return true;
    }

    virtual bool drag_data_delete_vfunc(const Gtk::TreeModel::Path &path)
    {
        Gtk::TreeModel::iterator it = get_iter(path);
        if( it )
            m_rearrange_signal((*it)[m_cols.m_id], m_new_parent_id);

        return true;
    }

private:
    Cols &m_cols;

    mutable Gtk::TreePath m_dragged_path;

    ObjectID m_new_parent_id;

    mutable RearrangeSignal m_rearrange_signal;
};



ExplorerController::ExplorerController(Gui &gui)
    : m_gui(gui), m_cols(new Cols)
{
    m_gui.get_builder()->get_widget("scene_browser", m_view);

    // Configure widgets
    m_model = DndTreeStore::create(*m_cols);
    m_model->rearrange_signal().connect(
        boost::bind(&ExplorerController::on_rearrange, this, _1, _2));

    m_view->set_search_column(m_cols->m_name);
    m_view->set_model(m_model);
    m_view->set_reorderable();

    // Add columns to the view
    Gtk::TreeView::Column *name_col =
        Gtk::manage( new Gtk::TreeView::Column("Name") );
    name_col->pack_start(m_cols->m_icon, false);
    name_col->pack_start(m_cols->m_name);
    m_view->append_column(*name_col);

    // Don't allow top level nodes to be selected
    m_selection = m_view->get_selection();
    m_selection->set_select_function(
        sigc::mem_fun(*this, &ExplorerController::select_handler) );
    // Allow multi-selection
    m_selection->set_mode(Gtk::SELECTION_MULTIPLE);
    m_view_selection_changed_conn = m_selection->signal_changed().connect(
        sigc::mem_fun(*this, &ExplorerController::on_view_selection_changed));

    // Connect slots
    m_gui.object_attached_signal().connect(
        boost::bind(&ExplorerController::object_attached_handler,
                    this, _1, _2, _3, _4));

    m_gui.object_detached_signal().connect(
        boost::bind(&ExplorerController::object_detached_handler, this, _1));

    m_gui.object_renamed_signal().connect(
        boost::bind(&ExplorerController::object_renamed_handler, this, _1, _2));

    m_gui.selection_changed_signal().connect(
        boost::bind(&ExplorerController::selection_changed_handler,
                    this, _1, _2));
}


ExplorerController::~ExplorerController()
{
    // Disconnect slots
    m_gui.object_attached_signal().disconnect(
        boost::bind(&ExplorerController::object_attached_handler,
                    this, _1, _2, _3, _4));

    m_gui.object_detached_signal().disconnect(
        boost::bind(&ExplorerController::object_detached_handler, this, _1));

    m_gui.object_renamed_signal().disconnect(
        boost::bind(&ExplorerController::object_renamed_handler, this, _1, _2));

    m_gui.selection_changed_signal().disconnect(
        boost::bind(&ExplorerController::selection_changed_handler,
                    this, _1, _2));
}


void ExplorerController::object_attached_handler(
    ObjectID id, ObjectID parent,
    std::string name, ObjectType type)
{
    if( parent == 0 )
    {
        // Add a top-level "Scene" row
        assert( m_tree_iters.empty() );
        Gtk::TreeModel::iterator it = m_model->append();
        Gtk::TreeModel::Row row = *it;
        row[m_cols->m_name] = "Scene";
        row[m_cols->m_id] = 0;
        m_tree_iters.insert( std::make_pair(id, it) );
    }
    else
    {
        // Add a new row as a child of the parent row
        TreeIters::iterator iter_it = m_tree_iters.find(parent);
        assert( iter_it != m_tree_iters.end() );
        Gtk::TreeModel::iterator parent_it = iter_it->second;

        Gtk::TreeModel::iterator it = m_model->append((*parent_it).children());
        Gtk::TreeModel::Row row = *it;
        row[m_cols->m_name] = name;
        row[m_cols->m_icon] = m_gui.get_icon(type);
        row[m_cols->m_id] = id;

        // Add an ObjectID to tree path entry, so we can find it again fast
        // given an ObjectID
        m_tree_iters.insert( std::make_pair(id, it) );
    }
}


void ExplorerController::object_detached_handler(ObjectID id)
{
    TreeIters::iterator iter_it = m_tree_iters.find(id);
    assert( iter_it != m_tree_iters.end() );

    Gtk::TreeModel::iterator it = iter_it->second;

    assert( !m_selection->is_selected(it) );

    m_model->erase(iter_it->second);

    // Remove id -> tree path mapping
    m_tree_iters.erase(iter_it);
}


void ExplorerController::object_renamed_handler(ObjectID id, std::string name)
{
    // Update the name in the model
    TreeIters::iterator iter_it = m_tree_iters.find(id);
    assert( iter_it != m_tree_iters.end() );

    Gtk::TreeModel::Row row = *iter_it->second;
    row[m_cols->m_name] = name;
}


void ExplorerController::selection_changed_handler(ObjectID id, bool selected)
{
    m_view_selection_changed_conn.block();

    TreeIters::iterator iter_it = m_tree_iters.find(id);
    if( iter_it == m_tree_iters.end() )
        return;

    Gtk::TreeModel::iterator it = iter_it->second;

    if( selected )
    {
        Gtk::TreePath tp = Gtk::TreePath(it);
        tp.up();
        m_view->expand_to_path(tp);
        m_selection->select(it);
    }
    else
    {
        m_selection->unselect(it);
    }

    m_view_selection_changed_conn.unblock();
}


bool ExplorerController::select_handler(
    const Glib::RefPtr<Gtk::TreeModel> &model,
    const Gtk::TreeModel::Path &path, bool)
{
    // Don't allow top level nodes to be selected
    return path.size() > 1;
}


void ExplorerController::on_view_selection_changed()
{
    std::set<ObjectID> selected;
    m_selection->selected_foreach_iter(
        sigc::bind(
            sigc::mem_fun(*this, &ExplorerController::selected_row_callback),
            sigc::ref(selected)) );

    m_gui.set_selection(selected);
}


void ExplorerController::selected_row_callback(
    const Gtk::TreeModel::iterator &it,
    std::set<ObjectID> &selected)
{
    selected.insert((*it)[m_cols->m_id]);
}


void ExplorerController::on_rearrange(ObjectID id, ObjectID new_parent_id)
{
    m_gui.server_post(
        boost::bind(
            &ExplorerController::ss_rearrange,
            this, _1, id, new_parent_id));
}


void ExplorerController::ss_rearrange(
    ServerData &sd, ObjectID id, ObjectID new_parent_id)
{
    SceneObject *obj = sd.m_scene->get_object(id);
    SceneObject *new_parent = (new_parent_id != 0) ?
        sd.m_scene->get_object(new_parent_id) : sd.m_scene->get_root();

    if( obj && new_parent )
    {
        try
        {
            obj->rearrange(new_parent, true);
        }
        catch(std::exception &e)
        {
            std::string error_str = e.what();
            m_gui.post(
                boost::bind(
                    &MainWindowController::display_message,
                    &m_gui.get_main_window_controller(),
                    Gtk::MESSAGE_ERROR, error_str));
        }
    }
}
