/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "FloatPropMediator.hh"
#include "../SceneObject.hh"
#include "../props/FloatPropBase.hh"

#include <cassert>
#include <limits>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


FloatPropMediator::FloatPropMediator(
    PropInspectorController &pi, SceneObject *obj,
    FloatPropBase *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0),
      m_display_factor(prop->display_factor()),
      m_hscale(0), m_spin_button(0), m_lbl(0)
{
    // Create and init widget
    post(
        boost::bind(&FloatPropMediator::create_widget, this,
                    m_obj->get_prop_name(m_prop_key),
                    any_cast<float>(m_prop->get(m_obj)),
                    m_prop->is_read_only(),
                    m_prop->get_min_value(), m_prop->get_max_value(),
                    m_prop->display_as_slider()));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&FloatPropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void FloatPropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(boost::bind(&FloatPropMediator::destroy_widget, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&FloatPropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void FloatPropMediator::create_widget(
    std::string prop_name,
    float val, float read_only,
    float min, float max,
    bool slider)
{
    assert( !m_hscale );
    assert( !m_spin_button );
    assert( !m_lbl );

    val *= m_display_factor;
    min *= m_display_factor;
    max *= m_display_factor;

    Gtk::Widget *widget;
    if( slider )
    {
        m_hscale = new Gtk::HScale();
        m_hscale->set_range(min, max);
        m_hscale->set_value(val);
        m_hscale->set_increments((max-min)/100, (max-min)/10);
        m_hscale->set_draw_value(false);
        m_widget_set_conn = m_hscale->signal_value_changed().connect(
            sigc::mem_fun(*this, &FloatPropMediator::on_widget_set));
        widget = m_hscale;
    }
    else
    {
        m_spin_button = new Gtk::SpinButton();
        m_spin_button->set_numeric(true);
        m_spin_button->set_range(min, max);
        m_spin_button->set_value(val);
        m_spin_button->set_width_chars(5);
        if( min == -std::numeric_limits<float>::max() ||
            max == std::numeric_limits<float>::max() )
            m_spin_button->set_increments(0.1, 1);
        else
            m_spin_button->set_increments((max-min)/100, (max-min)/10);
        m_spin_button->set_digits(2);
        m_widget_set_conn = m_spin_button->signal_value_changed().connect(
            sigc::mem_fun(*this, &FloatPropMediator::on_widget_set));
        widget = m_spin_button;
    }

    assert( widget );

    if( read_only )
        widget->set_sensitive(false);

    m_lbl = new Gtk::Label(prop_name + ":", 1.0, 0.5);

    add_prop_widgets(m_lbl, widget);
}


// Executed in the GUI thread
void FloatPropMediator::destroy_widget()
{
    assert( (m_hscale || m_spin_button) && !(m_hscale && m_spin_button) );
    assert( m_lbl );

    Gtk::Widget *widget;
    if( m_hscale )
    {
        widget = m_hscale;
    }
    else
    {
        assert( m_spin_button );
        widget = m_spin_button;
    }

    assert( widget );

    erase_prop_widgets(m_lbl, widget);
    delete widget;
    m_hscale = 0;
    m_spin_button = 0;
    delete m_lbl;
    m_lbl = 0;

    delete this;
}


// Executed in the server thread
void FloatPropMediator::on_prop_set()
{
    float val = any_cast<float>(m_prop->get(m_obj));
    post(boost::bind(&FloatPropMediator::update_widget, this, val));
}


// Executed in the GUI thread
void FloatPropMediator::update_widget(float val)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    val *= m_display_factor;

    m_widget_set_conn.block();

    if( m_hscale )
    {
        m_hscale->set_value(val);
    }
    else
    {
        m_spin_button->set_value(val);
    }

    m_widget_set_conn.unblock();
}


// Executed in the GUI thread
void FloatPropMediator::on_widget_set()
{
    ++m_n_updates_queued;

    float val;
    if( m_hscale )
    {
        val = m_hscale->get_value();
    }
    else
    {
        assert( m_spin_button );
        val = m_spin_button->get_value();
    }

    set_prop(m_object_id, m_prop_key, val/m_display_factor);
}
