/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "FrameLayoutController.hh"
#include "Frame.hh"
#include "Gui.hh"

#include <cassert>
#include <stdexcept>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


struct FrameLayoutController::Node
{
    Node()
        : m_parent(0), m_parent_box(0), m_widget(0), m_frame(0)
    {
        for( std::size_t i = 0; i < 2; ++i )
        {
            m_child[i] = 0;
            m_child_box[i] = 0;
        }
    }

    ~Node()
    {
        for( std::size_t i = 0; i < 2; ++i )
        {
            if( m_child[i] )
            {
                delete m_child[i];
                m_child[i] = 0;
                delete m_child_box[i];
                m_child_box[i] = 0;
            }
        }

        assert( m_widget );

        if( m_frame )
        {
            delete m_frame;
            m_frame = 0;
            m_widget = 0;
        }
        else
        {
            delete m_widget;
            m_widget = 0;
        }
    }

    Node *m_parent;

    Node *m_child[2];

    Gtk::VBox *m_child_box[2];

    Gtk::VBox *m_parent_box;

    Gtk::Widget *m_widget;

    Frame *m_frame;

    bool is_leaf() const { return !m_child[0]; }

    bool is_root() const { return !m_parent; }
};


FrameLayoutController::FrameLayoutController(Gui &gui)
    : m_gui(gui),
      m_root(0)
{
}


FrameLayoutController::~FrameLayoutController()
{
    m_root->m_parent_box->children().clear();
    delete m_root;
}


void FrameLayoutController::set_initial_frame(Frame *frame)
{
    if( m_root )
        throw std::logic_error(
            "set_initial_frame() must only be called once");

    m_root = new Node();
    m_root->m_frame = frame;
    m_root->m_widget = m_root->m_frame->get_frame_widget();

    m_gui.get_builder()->get_widget(
        "split_area_placeholder", m_root->m_parent_box);
    m_root->m_parent_box->add(*m_root->m_widget);

    m_frame_added_signal(frame);
}


FrameLayoutController::Node *
FrameLayoutController::find_node(const Frame *frame)
{
    return find_node_impl(frame, m_root);
}


FrameLayoutController::Node *
FrameLayoutController::find_node_impl(const Frame *frame, Node *node)
{
    if( node->m_frame == frame )
    {
        return node;
    }
    else if( node->is_leaf() )
    {
        return 0;
    }
    else
    {
        Node *tmp = find_node_impl(frame, node->m_child[0]);
        return ( tmp ) ? tmp : find_node_impl(frame, node->m_child[1]);
    }
}


void FrameLayoutController::split(
    const Frame *frame, SplitDirection dir, Frame *new_frame)
{
    assert( m_root );

    Node *node = find_node(frame);

    assert( node );
    assert( node->is_leaf() );
    assert( node->m_child_box[0] == 0 );
    assert( node->m_child_box[1] == 0 );

    // Remove the node's widget from its parent and create child boxes
    node->m_parent_box->children().clear();
    node->m_child_box[0] = new Gtk::VBox;
    node->m_child_box[1] = new Gtk::VBox;

    // Initialize the child nodes
    for( std::size_t i = 0; i < 2; ++i )
    {
        node->m_child[i] = new Node();
        node->m_child[i]->m_parent = node;
        node->m_child[i]->m_parent_box = node->m_child_box[i];
    }

    node->m_child[0]->m_frame = node->m_frame;
    node->m_child[1]->m_frame = new_frame;
    node->m_child[0]->m_widget = node->m_child[0]->m_frame->get_frame_widget();
    node->m_child[1]->m_widget = node->m_child[1]->m_frame->get_frame_widget();

    // Add the child widgets to the boxes
    for( std::size_t i = 0; i < 2; ++i )
    {
        node->m_child_box[i]->add(*node->m_child[i]->m_widget);
    }

    // Construct the new node widget, a paned
    Gtk::Paned *paned;
    if( dir == VSPLIT )
    {
        paned = new Gtk::HPaned();
        paned->set_position(node->m_parent_box->get_width()/2);
    }
    else
    {
        paned = new Gtk::VPaned();
        paned->set_position(node->m_parent_box->get_height()/2);
    }

    node->m_frame = 0;
    node->m_widget = paned;
    // Add the boxed child widgets to the paned
    paned->add1(*node->m_child_box[0]);
    paned->add2(*node->m_child_box[1]);
    // And add the paned to the parent box
    node->m_parent_box->add(*node->m_widget);
    paned->show_all();

    m_frame_added_signal(new_frame);
}


void FrameLayoutController::collapse(const Frame *frame, bool collapse_other)
{
    assert( m_root );

    Node *node = find_node(frame);

    assert( node );
    assert( node->is_leaf() );

    // We can't collapse the root node
    if( node->is_root() )
        return;

    if( collapse_other )
        collapse_child(node->m_parent, node->m_parent->m_child[0] == node);
    else
        collapse_child(node->m_parent, node->m_parent->m_child[0] != node);
}


void FrameLayoutController::collapse_child(Node *node, bool second)
{
    assert( !node->is_leaf() );

    // Remove the children widgets from our widget
    node->m_child_box[0]->children().clear();
    node->m_child_box[1]->children().clear();
    // Remove the old parent widget from its box
    node->m_parent_box->children().clear();

    std::size_t keep_idx = !second;
    Node *kept = node->m_child[keep_idx];

    // We need to call this before actually deleting the node, since doing so
    // will delete the frame as well
    m_frame_removed_signal(node->m_child[!keep_idx]->m_frame);

    node->m_child[keep_idx] = 0;

    // Update parent pointer and box on kept subtree
    kept->m_parent = node->m_parent;
    kept->m_parent_box = node->m_parent_box;

    // Add the subtree widget to its parent box
    kept->m_parent_box->add(*kept->m_widget);

    if( !node->is_root() )
    {
        // Update the tree structure
        Node *parent = node->m_parent;
        if( parent->m_child[0] == node )
            parent->m_child[0] = kept;
        else
            parent->m_child[1] = kept;

        delete node;
    }
    else
    {
        // We're collapsing a child to the root node
        delete m_root;
        m_root = kept;
    }
}
