/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Gui.hh"
#include "MainWindowController.hh"
#include "FrameLayoutController.hh"
#include "ExplorerController.hh"
#include "PropInspectorController.hh"
#include "LogViewController.hh"
#include "PbarPlayerController.hh"
#include "ToolController.hh"
#include "RendererRelay.hh"
#include "SceneViewFrame.hh"
#include "../Config.hh"
#include "../Server.hh"
#include "../SceneObject.hh"
#include "../CameraObject.hh"
#include "../SceneTree.hh"
#include "../Path.hh"
#include "../FsToolbox.hh"
#include "../ObjectTypes.hh"

#include <sstream>
#include <cassert>
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/xpressive/xpressive.hpp>
#include <boost/lexical_cast.hpp>
#include <algorithm>


using namespace peekabot;
using namespace peekabot::gui;



Gui::Gui(Config &config, Server &server)
    : m_config(config),
      m_server(server),
      m_renderer(new RendererRelay(*this)),
      m_gl_config(
          Gdk::GL::Config::create(
              Gdk::GL::MODE_RGBA |
              Gdk::GL::MODE_DEPTH |
              Gdk::GL::MODE_STENCIL |
              Gdk::GL::MODE_DOUBLE)),
      m_dummy_gl_area(m_gl_config, true)
{
    m_gui_work_queue.set_processing_freq(
        m_config.get<float>("gui.update_freq", 10));

    boost::filesystem::path icon_dir =
        m_config.get<boost::filesystem::path>("resources.share_path") / "icons";
    if( !boost::filesystem::exists(icon_dir) )
        throw std::runtime_error("Icon directory not found");
    load_stock_icons(icon_dir);

    // Create the group icon from the stock directory icon
    Gtk::IconTheme::add_builtin_icon(
        "peekabot-group", 16,
        Gtk::IconTheme::get_default()->load_icon("gtk-directory", 16));

    // Load the GUI from the Glade file
    boost::filesystem::path ui_path =
        m_config.get<boost::filesystem::path>("resources.share_path") /
        "peekabot.ui";
    m_builder = Gtk::Builder::create_from_file(ui_path.string());

    m_main_window_controller.reset(new MainWindowController(*this));
    m_frame_layout_controller.reset(new FrameLayoutController(*this));
    m_explorer_controller.reset(new ExplorerController(*this));
    m_prop_inspector_controller.reset(new PropInspectorController(*this));
    m_pbar_player_controller.reset(new PbarPlayerController(*this));
    m_log_view_controller.reset(new LogViewController(*this));
    m_tool_controller.reset(new ToolController(*this));

    Gtk::VBox *main_area_vbox = 0;
    m_builder->get_widget("main_window", m_main_window);
    m_builder->get_widget("main_area_vbox", main_area_vbox);

    main_area_vbox->pack_start(m_dummy_gl_area);
    m_dummy_gl_area.signal_map().connect(
        sigc::mem_fun(*this, &Gui::on_dummy_gl_area_mapped));
    m_dummy_gl_area.show();

    m_main_window->set_sensitive(false);
    m_main_window->show();

    // Populate the GUI with date from the scene
    server_post(boost::bind(&Gui::init_gui, this, _1));
}


Gui::~Gui()
{
}


void Gui::on_dummy_gl_area_mapped()
{
    // Create the OpenGL context
    create_gl_context();

    // Initialize the renderer
    init_renderer();

    m_frame_layout_controller->frame_added_signal().connect(
        boost::bind(&Gui::on_frame_added, this, _1));

    m_frame_layout_controller->frame_removed_signal().connect(
        boost::bind(&Gui::on_frame_removed, this, _1));

    // Set the initial frame
    m_frame_layout_controller->set_initial_frame(
        new SceneViewFrame(*m_frame_layout_controller));

    // Start the GUI work queue
    m_gui_work_queue.start();

    m_dummy_gl_area.hide();
    m_main_window->set_sensitive(true);
}


void Gui::create_gl_context()
{
    Glib::RefPtr<Gdk::GL::Window> gl_window = m_dummy_gl_area.get_gl_window();
    assert( gl_window );

    m_gl_context = Gdk::GL::Context::create(gl_window, true);
    assert( m_gl_context );
    m_dummy_gl_context = Gdk::GL::Context::create(gl_window, true);
    assert( m_dummy_gl_context );

    if( !m_gl_context->is_direct() )
        m_main_window_controller->display_message(
            Gtk::MESSAGE_WARNING, "OpenGL direct rendering disabled");
}


Config &Gui::get_config()
{
    return m_config;
}


const Config &Gui::get_config() const
{
    return m_config;
}


Server &Gui::get_server()
{
    return m_server;
}


const Server &Gui::get_server() const
{
    return m_server;
}


void Gui::server_post(const boost::function<void (ServerData &)> &handler)
{
    m_server.post(handler);
}


void Gui::run()
{
    Gtk::Main::run(*m_main_window);
}


const Glib::RefPtr<Gdk::GL::Config> &Gui::get_gl_config()
{
    return m_gl_config;
}


const Glib::RefPtr<Gdk::GL::Context> &Gui::get_gl_context()
{
    return m_gl_context;
}


const Glib::RefPtr<Gdk::GL::Context> &Gui::get_dummy_gl_context()
{
    return m_dummy_gl_context;
}


void Gui::gl_begin()
{
    if( !m_dummy_gl_area.get_gl_window()->gl_begin(m_gl_context) )
        throw std::runtime_error("gl_begin() failed");
}


void Gui::gl_end()
{
    m_dummy_gl_area.get_gl_window()->gl_end();

    // HACK! This should be a no-op but for some reason inserting this code
    // makes the program not mysteriously crash on my Intel card with
    // Fedora 14.
    m_dummy_gl_area.get_gl_window()->make_current(m_dummy_gl_context);
}


RendererRelay *Gui::get_renderer()
{
    return m_renderer.get();
}


const Glib::RefPtr<Gtk::Builder> &Gui::get_builder()
{
    return m_builder;
}


const Gui::Cameras &Gui::get_cameras() const
{
    return m_cameras;
}


void Gui::init_gui(ServerData &sd)
{
    // Registers slots, extract a list of all camera, etc.
    on_object_attached(sd.m_scene->get_root());
}


void Gui::init_renderer()
{
    try
    {
        m_renderer->init();
    }
    catch(std::exception &e)
    {
        std::stringstream ss;
        Gtk::MessageDialog dialog(
            "Failed to initialize renderer", false, Gtk::MESSAGE_ERROR);
        dialog.set_secondary_text(e.what());
        dialog.run();

        throw;
    }

    m_renderer->set_max_fps(m_config.get<float>("gui.max_fps", 40));

    m_renderer->set_background_color(
        m_config.get<RGBColor>(
            "gui.background_color", RGBColor(1, 1, 1)));

    m_renderer->set_selected_color(
        m_config.get<RGBColor>(
            "gui.selected_color", RGBColor(0, 0.3, 1)));

    m_renderer->set_ancestor_selected_color(
        m_config.get<RGBColor>(
            "gui.ancestor_selected_color", RGBColor(0.5, 0.75, 1)));
}


// NOTE: EXECUTED IN THE SERVER
void Gui::on_object_attached(SceneObject *obj)
{
    // Note: Order matters here, this should be calling before recursing since
    // we want the signal to emit for parents before it emits for their
    // children
    SceneObject *parent = obj->get_parent();
    post( boost::bind(
              &Gui::object_attached_handler, this,
              obj->get_object_id(),
              parent ? parent->get_object_id() : 0,
              obj->get_name(), obj->get_object_type()) );

    // Register triggers
    obj->child_attached_signal().connect(
        boost::bind(&Gui::on_object_attached, this, _1));

    obj->detached_signal().connect(
        boost::bind(&Gui::on_object_detached, this, obj));

    obj->selected_set_signal().connect(
        boost::bind(&Gui::on_object_selected, this, obj));

    obj->name_set_signal().connect(
        boost::bind(&Gui::on_object_name_set, this, obj));

    if( obj->is_selected() )
        on_object_selected(obj);

    for( SceneObject::ChildIterator it = obj->begin(); it != obj->end(); ++it )
    {
        on_object_attached(*it);
    }
}


// NOTE: EXECUTED IN THE SERVER
void Gui::on_object_detached(SceneObject *obj)
{
    // Deregister triggers
    obj->child_attached_signal().disconnect(
        boost::bind(&Gui::on_object_attached, this, _1));

    obj->detached_signal().disconnect(
        boost::bind(&Gui::on_object_detached, this, obj));

    obj->selected_set_signal().disconnect(
        boost::bind(&Gui::on_object_selected, this, obj));

    obj->name_set_signal().disconnect(
        boost::bind(&Gui::on_object_name_set, this, obj));

    if( obj->is_selected() )
    {
        obj->set_selected(false);
        on_object_selected(obj);
    }

    for( SceneObject::ChildIterator it = obj->begin(); it != obj->end(); ++it )
    {
        on_object_detached(*it);
    }

    // Note: Order matters here! This need to be called last so that the
    // signal triggers for children before their parents
    post( boost::bind(&Gui::object_detached_handler,
                      this, obj->get_object_id()) );
}


// NOTE: EXECUTED IN THE SERVER
void Gui::on_object_selected(SceneObject *obj)
{
    bool selected = obj->is_selected();
    if( selected )
    {
        m_ss_selection.insert(obj);
    }
    else
    {
        m_ss_selection.erase(obj);
    }

    m_ss_selection_changed_signal(obj, selected);

    // Update the m_selection set used in the GUI thread
    post(boost::bind(
             &Gui::on_object_selected_set, this,
             obj->get_object_id(), selected));
}


// NOTE: EXECUTED IN THE SERVER
void Gui::on_object_name_set(SceneObject *obj)
{
    post( boost::bind(
              &Gui::object_name_set_handler, this,
              obj->get_object_id(), obj->get_name()) );
}


void Gui::save_screenshot(
    int w, int h,
    ObjectID cam_id,
    const bool *layers,
    const std::string &filename)
{
    assert( w > 0 );
    assert( h > 0 );

    if( !m_renderer )
        throw std::runtime_error("Renderer not initialized");

    Glib::RefPtr<Gdk::Pixmap> pixmap = Gdk::Pixmap::create(
        m_main_window->get_window(), w, h, m_gl_config->get_depth());

    Glib::RefPtr<Gdk::GL::Pixmap> glpixmap =
        Gdk::GL::Pixmap::create(m_gl_config, pixmap);

    if( !glpixmap->gl_begin(m_gl_context) )
        return;

    try
    {
        m_renderer->render(w, h, cam_id, layers);
    }
    catch(...)
    {
    }

    if( glpixmap->is_double_buffered() )
        glpixmap->swap_buffers();
    else
        glFlush();

    glpixmap->gl_end();

    // HACK! This should be a no-op but for some reason inserting this code
    // makes the program not mysteriously crash on my Intel card with
    // Fedora 14.
    glpixmap->make_current(m_dummy_gl_context);

    Glib::RefPtr<Gdk::Pixbuf> pixbuf =
        Gdk::Pixbuf::create(
            Glib::RefPtr<Gdk::Drawable>::cast_dynamic(pixmap),
            0, 0, w, h);
    pixbuf->save(filename, "png");
}


void Gui::object_attached_handler(
    ObjectID id, ObjectID parent,
    std::string name, ObjectType type)
{
    if( type == CAMERA_OBJECT )
    {
        m_cameras[id] = name;
    }

    m_object_attached_signal(id, parent, name, type);
}


void Gui::object_detached_handler(ObjectID id)
{
    m_cameras.erase(id);

    m_object_detached_signal(id);
}


const std::set<SceneObject *> &Gui::ss_get_selection() const
{
    return m_ss_selection;
}


void Gui::set_selected(ObjectID id, bool selected)
{
    server_post(
        boost::bind(&Gui::set_selected_, this, _1, id, selected));
}


void Gui::toggle_select(ObjectID id)
{
    set_selected(id, !m_selection.count(id));
}


const Gui::Selection &Gui::get_selection() const
{
    return m_selection;
}


void Gui::set_selection(const Selection &selection)
{
    // Calculate the objects to select and deselect:
    //   to be selected = new_selection - old_selection ^ new_selection
    //   to be deselected = old_selection - old_selection ^ new_selection
    std::set<ObjectID> intersection;
    std::set_intersection(
        m_selection.begin(), m_selection.end(),
        selection.begin(), selection.end(),
        std::inserter(intersection, intersection.begin()));

    std::set<ObjectID> select;
    std::set_difference(
        selection.begin(), selection.end(),
        intersection.begin(), intersection.end(),
        std::inserter(select, select.begin()));

    std::set<ObjectID> deselect;
    std::set_difference(
        m_selection.begin(), m_selection.end(),
        intersection.begin(), intersection.end(),
        std::inserter(deselect, deselect.begin()));

    server_post(
        boost::bind(&Gui::update_selected, this, _1,
                    select, deselect));
}


void Gui::clear_selection()
{
    server_post(
        boost::bind(&Gui::update_selected, this, _1,
                    Selection(), m_selection));
}


void Gui::set_selected_(
    ServerData &sd, ObjectID object_id, bool select)
{
    SceneObject *obj = sd.m_scene->get_object(object_id);
    if( obj )
    {
        obj->set_selected(select);
    }
}


void Gui::update_selected(
    ServerData &sd, std::set<ObjectID> select, std::set<ObjectID> deselect)
{
    BOOST_FOREACH( ObjectID id, select )
    {
        SceneObject *obj = sd.m_scene->get_object(id);
        if( obj )
        {
            obj->set_selected(true);
        }
    }

    BOOST_FOREACH( ObjectID id, deselect )
    {
        SceneObject *obj = sd.m_scene->get_object(id);
        if( obj )
        {
            obj->set_selected(false);
        }
    }
}


void Gui::on_object_selected_set(ObjectID id, bool selected)
{
    if( selected )
    {
        if( m_selection.insert(id).second )
            m_selection_changed_signal(id, true);
    }
    else
    {
        if( m_selection.erase(id) > 0 )
            m_selection_changed_signal(id, false);
    }
}


void Gui::object_name_set_handler(ObjectID id, std::string name)
{
    Cameras::iterator it = m_cameras.find(id);
    if( it != m_cameras.end() )
    {
        it->second = name;
    }

    m_object_renamed_signal(id, name);
}


Glib::RefPtr<Gdk::Pixbuf> Gui::get_icon(ObjectType type)
{
    if( type == SPHERE_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-sphere", 16);
    else if( type == CUBE_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-cube", 16);
    else if( type == CYLINDER_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-cylinder", 16);
    else if( type == CIRCLE_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-circle", 16);
    else if( type == POLYGON_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-polygon", 16);
    else if( type == LINE_CLOUD_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-line-cloud", 16);
    else if( type == POINT_CLOUD_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-point-cloud", 16);
    else if( type == GRID_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-grid", 16);
    else if( type == LABEL_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-label", 16);
    else if( type == CAMERA_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-camera", 16);
    else if( type == OG2D_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-og2d", 16);
    else if( type == OG3D_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-og3d", 16);
    else if( type == TRI_MESH_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-tri-mesh", 16);
    else if( type == GROUP_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-group", 16);
    else if( type == SLIDER_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-dof", 16);
    else if( type == HINGE_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-dof", 16);
    else if( type == POLYLINE_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-polyline", 16);
    else if( type == MODEL_OBJECT )
        return Gtk::IconTheme::get_default()->load_icon("peekabot-model", 16);
    else
        return Glib::RefPtr<Gdk::Pixbuf>();
}


MainWindowController &Gui::get_main_window_controller()
{
    return *m_main_window_controller;
}


ExplorerController &Gui::get_explorer_controller()
{
    return *m_explorer_controller;
}


PropInspectorController &Gui::get_prop_inspector_controller()
{
    return *m_prop_inspector_controller;
}


FrameLayoutController &Gui::get_frame_layout_controller()
{
    return *m_frame_layout_controller;
}


LogViewController &Gui::get_log_view_controller()
{
    return *m_log_view_controller;
}


PbarPlayerController &Gui::get_pbar_player_controller()
{
    return *m_pbar_player_controller;
}


ToolController &Gui::get_tool_controller()
{
    return *m_tool_controller;
}


void Gui::load_stock_icons(const boost::filesystem::path &dir)
{
    if( !boost::filesystem::exists(dir) ||
        !boost::filesystem::is_directory(dir) )
        return;

    boost::filesystem::directory_iterator end_it;
    for( boost::filesystem::directory_iterator it(dir); it != end_it; ++it )
    {
        if( boost::filesystem::is_directory(it->status()) )
        {
            load_stock_icons(it->path());
        }
        else
        {
            static const boost::xpressive::sregex re =
                boost::xpressive::sregex::compile("((?:\\w|-)+)-(\\d+)\\.png");

            boost::xpressive::smatch what;
            std::string leaf = it->path().leaf();

            if( boost::xpressive::regex_match(leaf, what, re) )
            {
                Gtk::IconTheme::add_builtin_icon(
                    "peekabot-" + what[1],
                    boost::lexical_cast<int>(what[2]),
                    Gdk::Pixbuf::create_from_file(it->path().string()));
            }
        }
    }
}


void Gui::redraw_scene_views()
{
    BOOST_FOREACH( SceneViewFrame *svf, m_scene_view_frames )
    {
        svf->redraw_scene_view();
    }
}


void Gui::on_frame_added(Frame *frame)
{
    SceneViewFrame *svf = dynamic_cast<SceneViewFrame *>(frame);
    if( svf )
        m_scene_view_frames.insert(svf);
}


void Gui::on_frame_removed(Frame *frame)
{
    SceneViewFrame *svf = dynamic_cast<SceneViewFrame *>(frame);
    if( svf )
        m_scene_view_frames.erase(svf);
}
