/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GUI_GUI_HH_INCLUDED
#define PEEKABOT_GUI_GUI_HH_INCLUDED


#include "../Types.hh"
#include "WorkQueue.hh"

#include <map>
#include <set>
#include <GL/glew.h>
#include <gtkmm.h>
#include <gtkglmm.h>
#include <boost/utility.hpp>
#include <boost/function.hpp>
#include <boost/signals2.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>


namespace peekabot
{
    class Config;
    class Server;
    class ServerData;
    class SceneObject;

    namespace gui
    {
        class RendererRelay;
        class MainWindowController;
        class ExplorerController;
        class PropInspectorController;
        class FrameLayoutController;
        class LogViewController;
        class PbarPlayerController;
        class ToolController;
        class Frame;
        class SceneViewFrame;

        /**
         * \brief The main GUI class which is instanced to create a GUI.
         *
         * \pre run() must be called in the thread that created the Gui
         * instance.
         *
         * Besides orchestrating the different components of the GUI it
         * provides various functionality that makes the life of said
         * components easier:
         *
         * \li Various GUI-side signals for objects in the scene (i.e. signals
         * executed in the GUI thread).
         * \li A work-queue where you can post tasks to be executed in the GUI
         * thread from any other thread (typically the server thread). See
         * post().
         * \li Functionality for getting and manipulating the current
         * selection (from within the GUI thread).
         */
        class Gui : public boost::noncopyable
        {
        public:
            typedef std::map<ObjectID, std::string> Cameras;

            Gui(Config &config, Server &server);

            ~Gui();

            Config &get_config();

            const Config &get_config() const;

            Server &get_server();

            const Server &get_server() const;

            inline void post(const boost::function<void ()> &handler)
            {
                m_gui_work_queue.post(handler);
            }

            /**
             * \brief Convenience method to post a task to be executed in the
             * server thread.
             *
             * This method is equivalent to \c get_server().post().
             */
            void server_post(
                const boost::function<void (ServerData &)> &handler);

            /// \name GUI-side methods
            /// @{

            void run();

            const Glib::RefPtr<Gdk::GL::Config> &get_gl_config();

            const Glib::RefPtr<Gdk::GL::Context> &get_gl_context();

            const Glib::RefPtr<Gdk::GL::Context> &get_dummy_gl_context();

            void gl_begin();

            void gl_end();

            MainWindowController &get_main_window_controller();

            ExplorerController &get_explorer_controller();

            PropInspectorController &get_prop_inspector_controller();

            FrameLayoutController &get_frame_layout_controller();

            LogViewController &get_log_view_controller();

            PbarPlayerController &get_pbar_player_controller();

            ToolController &get_tool_controller();

            RendererRelay *get_renderer();

            const Glib::RefPtr<Gtk::Builder> &get_builder();

            const Cameras &get_cameras() const;

            void save_screenshot(
                int w, int h,
                ObjectID cam_id,
                const bool *layers,
                const std::string &filename);

            Glib::RefPtr<Gdk::Pixbuf> get_icon(ObjectType type);

            typedef std::set<ObjectID> Selection;

            void set_selected(ObjectID id, bool selected);

            void toggle_select(ObjectID id);

            const Selection &get_selection() const;

            void set_selection(const Selection &selection);

            void clear_selection();

            void redraw_scene_views();

            /// @{
            /// \name Server-side methods
            /// @{

            const std::set<SceneObject *> &ss_get_selection() const;

            /// @}
            /// \name GUI-side signals
            /// @{

            typedef boost::signals2::signal<void (
                ObjectID id, bool added)> SelectionChangedSignal;

            typedef boost::signals2::signal<void (
                ObjectID id, ObjectID parent_id,
                const std::string &name, ObjectType type)> ObjectAttachedSignal;

            typedef boost::signals2::signal<void (
                ObjectID id)> ObjectDetachedSignal;

            typedef boost::signals2::signal<void (
                ObjectID id, std::string)> ObjectRenamedSignal;

            inline SelectionChangedSignal &selection_changed_signal()
            {
                return m_selection_changed_signal;
            }

            inline ObjectAttachedSignal &object_attached_signal()
            {
                return m_object_attached_signal;
            }

            inline ObjectDetachedSignal &object_detached_signal()
            {
                return m_object_detached_signal;
            }

            inline ObjectRenamedSignal &object_renamed_signal()
            {
                return m_object_renamed_signal;
            }

            /// @}
            /// \name Server-side signals
            /// @{

            typedef boost::signals2::signal<void (
                SceneObject *, bool selected)> SsSelectionChangedSignal;

            inline SsSelectionChangedSignal &ss_selection_changed_signal()
            {
                return m_ss_selection_changed_signal;
            }

            /// @}

        private:
            void on_dummy_gl_area_mapped();

            void create_gl_context();

            void init_renderer();

            void object_attached_handler(
                ObjectID id, ObjectID parent,
                std::string name, ObjectType type);

            void object_detached_handler(ObjectID id);

            void on_object_selected_set(ObjectID id, bool selected);

            void object_name_set_handler(ObjectID id, std::string name);

            void load_stock_icons(const boost::filesystem::path &dir);

            void on_frame_added(Frame *frame);

            void on_frame_removed(Frame *frame);

            /// \name Methods executed in the server thread
            /// @{

            void init_gui(ServerData &sd);

            void on_object_attached(SceneObject *obj);

            void on_object_detached(SceneObject *obj);

            void on_object_selected(SceneObject *obj);

            void on_object_name_set(SceneObject *obj);

            void set_selected_(
                ServerData &sd, ObjectID object_id, bool select);

            void update_selected(
                ServerData &sd,
                std::set<ObjectID> select,
                std::set<ObjectID> deselect);

            /// @}

        private:
            SelectionChangedSignal m_selection_changed_signal;
            ObjectAttachedSignal m_object_attached_signal;
            ObjectDetachedSignal m_object_detached_signal;
            ObjectRenamedSignal m_object_renamed_signal;
            SsSelectionChangedSignal m_ss_selection_changed_signal;

            Config &m_config;

            Server &m_server;

            boost::scoped_ptr<RendererRelay> m_renderer;

            WorkQueue m_gui_work_queue;

            Glib::RefPtr<Gtk::Builder> m_builder;

            Gtk::Window *m_main_window;

            // Misc

            Glib::RefPtr<Gdk::GL::Config> m_gl_config;

            Gtk::GL::DrawingArea m_dummy_gl_area;

            Glib::RefPtr<Gdk::GL::Context> m_gl_context;
            Glib::RefPtr<Gdk::GL::Context> m_dummy_gl_context;

            Cameras m_cameras;

            Selection m_selection;

            // Access only in the server thread!
            std::set<SceneObject *> m_ss_selection;


            boost::scoped_ptr<MainWindowController> m_main_window_controller;
            boost::scoped_ptr<FrameLayoutController> m_frame_layout_controller;
            boost::scoped_ptr<ExplorerController> m_explorer_controller;
            boost::scoped_ptr<PropInspectorController> m_prop_inspector_controller;
            boost::scoped_ptr<LogViewController> m_log_view_controller;
            boost::scoped_ptr<PbarPlayerController> m_pbar_player_controller;
            boost::scoped_ptr<ToolController> m_tool_controller;

            std::set<SceneViewFrame *> m_scene_view_frames;
        };
    }
}


#endif // PEEKABOT_GUI_GUI_HH_INCLUDED
