/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MainWindowController.hh"
#include "PreferencesDialog.hh"
#include "Gui.hh"
#include "RendererRelay.hh"
#include "../Server.hh"
#include "../Config.hh"
#include "../SceneFileLoader.hh"
#include "../SceneTree.hh"
#include "../SceneObject.hh"
#include "../CameraObject.hh"
#include "../Version.hh"
#include "../Path.hh"
#include "../FsToolbox.hh"
#include "../ObjectTypes.hh"

#include <cassert>
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <sstream>


using namespace peekabot;
using namespace peekabot::gui;


MainWindowController::MainWindowController(Gui &gui)
    : m_gui(gui),
      m_preferences_dialog(new PreferencesDialog(m_gui))
{
    setup_widgets();
    setup_actions();
    setup_infobar();

    m_gui.object_attached_signal().connect(
        boost::bind(&MainWindowController::add_camera, this,
                    _1, _2, _3, _4));

    m_gui.object_detached_signal().connect(
        boost::bind(&MainWindowController::remove_camera, this, _1));

    m_gui.selection_changed_signal().connect(
        boost::bind(&MainWindowController::on_selection_changed, this));

    m_fps_updated_conn = m_gui.get_renderer()->fps_updated_signal().connect(
        boost::bind(&MainWindowController::on_fps_updated, this, _1));

    m_main_window->maximize();
}


MainWindowController::~MainWindowController()
{
}


void MainWindowController::setup_widgets()
{
    const Glib::RefPtr<Gtk::Builder> &builder = m_gui.get_builder();

    builder->get_widget("main_window", m_main_window);
    builder->get_widget("main_paned", m_main_paned);
    builder->get_widget("sidebar_paned", m_sidebar_paned);
    builder->get_widget("sidebar_alignment", m_sidebar_alignment);
    builder->get_widget("about_dialog", m_about_dialog);
    builder->get_widget("load_scene_dialog", m_load_scene_dialog);
    builder->get_widget("toolbar", m_toolbar);
    builder->get_widget("menubar", m_menubar);
    builder->get_widget("statusbar", m_statusbar);
    builder->get_widget("main_area_vbox", m_main_area_vbox);
    builder->get_widget("screenshot_dialog", m_screenshot_dialog);
    builder->get_widget(
        "screenshot_w_spinbutton", m_screenshot_w_spinbutton);
    builder->get_widget(
        "screenshot_h_spinbutton", m_screenshot_h_spinbutton);
    builder->get_widget(
        "screenshot_layer_table", m_screenshot_layer_table);
    builder->get_widget(
        "screenshot_filename_entry", m_screenshot_filename_entry);
    builder->get_widget(
        "screenshot_cameras", m_screenshot_cameras);
    builder->get_widget("screenshot_save_dialog", m_screenshot_save_dialog);
    builder->get_widget(
        "screenshot_browse_button", m_screenshot_browse_button);
    builder->get_widget("fps_label", m_fps_label);

    m_about_dialog->set_version(PEEKABOT_VERSION_STRING);
    m_about_dialog->signal_response().connect(
        sigc::mem_fun(*this, &MainWindowController::on_about_response));

    m_load_scene_dialog->signal_response().connect(
        sigc::mem_fun(*this, &MainWindowController::on_load_scene_response));

    m_screenshot_dialog->signal_response().connect(
        sigc::mem_fun(*this, &MainWindowController::on_screenshot_dialog_response));

    m_screenshot_w_spinbutton->set_range(1, 2560);
    m_screenshot_w_spinbutton->set_value(1280);
    m_screenshot_w_spinbutton->set_increments(1, 10);

    m_screenshot_h_spinbutton->set_range(1, 2560);
    m_screenshot_h_spinbutton->set_value(1024);
    m_screenshot_h_spinbutton->set_increments(1, 10);

    m_screenshot_cameras_model =
        Gtk::ListStore::create(m_screenshot_camera_cols);
    m_screenshot_cameras_model->set_sort_column(0, Gtk::SORT_ASCENDING);
    m_screenshot_cameras->set_model(m_screenshot_cameras_model);
    m_screenshot_cameras->pack_start(m_screenshot_camera_cols.m_name);

    m_screenshot_save_dialog->signal_response().connect(
        sigc::mem_fun(*this, &MainWindowController::on_screenshot_save_response));

    m_screenshot_browse_button->signal_clicked().connect(
        sigc::mem_fun(*m_screenshot_save_dialog,
                      &Gtk::FileChooserDialog::show));
}


void MainWindowController::setup_actions()
{
    const Glib::RefPtr<Gtk::Builder> &builder = m_gui.get_builder();

    m_about_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("about_action"));
    m_about_action->signal_activate().connect(
            sigc::mem_fun(*this, &MainWindowController::on_about));

    m_quit_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("quit_action"));
    m_quit_action->signal_activate().connect(
            sigc::mem_fun(*this, &MainWindowController::on_quit));

    m_load_scene_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("load_scene_action"));
    m_load_scene_action->signal_activate().connect(
            sigc::mem_fun(*this, &MainWindowController::on_load_scene));

    m_clear_scene_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("clear_scene_action"));
    m_clear_scene_action->signal_activate().connect(
            sigc::mem_fun(*this, &MainWindowController::on_clear_scene));

    m_add_camera_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("add_camera_action"));
    m_add_camera_action->signal_activate().connect(
            sigc::mem_fun(*this, &MainWindowController::on_add_camera));

    m_fullscreen_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        builder->get_object("fullscreen_action"));
    m_fullscreen_action->signal_toggled().connect(
            sigc::mem_fun(*this, &MainWindowController::on_fullscreen));

    m_sidebar_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        builder->get_object("sidebar_action"));
    m_sidebar_action->signal_toggled().connect(
        sigc::mem_fun(*this, &MainWindowController::on_toggle_sidebar));

    m_toggle_toolbar_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        builder->get_object("toggle_toolbar_action"));
    m_toggle_toolbar_action->signal_toggled().connect(
        sigc::mem_fun(*this, &MainWindowController::on_toggle_toolbar));

    m_toggle_statusbar_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        builder->get_object("toggle_statusbar_action"));
    m_toggle_statusbar_action->signal_toggled().connect(
        sigc::mem_fun(*this, &MainWindowController::on_toggle_statusbar));

    m_delete_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        builder->get_object("delete_action"));
    m_delete_action->signal_activate().connect(
        sigc::mem_fun(*this, &MainWindowController::on_delete));
}


void MainWindowController::setup_infobar()
{
    m_main_area_vbox->pack_start(m_infobar, Gtk::PACK_SHRINK);
    m_main_area_vbox->reorder_child(m_infobar, 1);

    Gtk::Container *infobar_container =
        dynamic_cast<Gtk::Container *>(m_infobar.get_content_area());
    if( infobar_container )
        infobar_container->add(m_infobar_hbox);
    m_infobar_hbox.pack_start(m_infobar_icon, Gtk::PACK_SHRINK);
    m_infobar_hbox.pack_start(m_infobar_lbl);
    m_infobar.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
    m_infobar_hbox.set_spacing(5);
    m_infobar.show_all();
    m_infobar.hide();
    m_infobar.signal_response().connect(
        sigc::mem_fun(*this, &MainWindowController::on_infobar_response));
}


void MainWindowController::add_camera(
    ObjectID id, ObjectID parent_id,
    std::string name, ObjectType type)
{
    if( type != CAMERA_OBJECT )
        return;

    // Update the camera combo box in the screenshot dialog
    bool was_empty = m_screenshot_cameras_model->children().size() == 0;

    Gtk::TreeModel::Row row = *(m_screenshot_cameras_model->append());
    row[m_screenshot_camera_cols.m_id] = id;
    row[m_screenshot_camera_cols.m_name] = name;

    if( was_empty )
        m_screenshot_cameras->set_active(0);
}


void MainWindowController::remove_camera(ObjectID id)
{
    // Update the camera combo box in the screenshot dialog
    Gtk::TreeModel::Children children = m_screenshot_cameras_model->children();
    for( Gtk::TreeModel::iterator it = children.begin(); it != children.end(); ++it )
    {
        Gtk::TreeModel::Row row = *it;
        if( row[m_screenshot_camera_cols.m_id] == id )
        {
            m_screenshot_cameras_model->erase(it);
            break;
        }
    }

    Gtk::TreeModel::iterator it = m_screenshot_cameras->get_active();
    if( !it )
        m_screenshot_cameras->set_active(0);
}


void MainWindowController::on_about()
{
    m_about_dialog->show();
}


void MainWindowController::on_about_response(int)
{
    m_about_dialog->hide();
}


void MainWindowController::on_load_scene()
{
    m_load_scene_dialog->show();
}


void MainWindowController::on_load_scene_response(int response)
{
    m_load_scene_dialog->hide();
    if( response == Gtk::RESPONSE_OK )
    {
        m_gui.server_post(
            boost::bind(&MainWindowController::load_scene, this, _1,
                        m_load_scene_dialog->get_filename()));
    }
}


void MainWindowController::on_clear_scene()
{
    m_gui.server_post(boost::bind(&MainWindowController::clear_scene, this, _1));
}


void MainWindowController::clear_scene(ServerData &sd)
{
    SceneObject *root = sd.m_scene->get_root();
    for( SceneObject::ChildIterator it = root->begin(); it != root->end(); )
    {
        SceneObject *obj = *it;
        ++it;
        if( obj->get_name() != "default_camera" )
        {
            obj->detach();
            delete obj;
        }
    }

    Path path(
        "default_scene.xml",
        boost::shared_ptr<UploadCache>(),
        m_gui.get_config().get<std::list<boost::filesystem::path> >("data_paths"));

    try
    {
        std::vector<SceneObject *> objs = SceneFileLoader::load_scene(
            m_gui.get_config(), path);

        // TODO: fix potential memory leaks here
        for( std::size_t i = 0; i < objs.size(); ++i )
            root->attach(objs[i], peekabot::AUTO_ENUMERATE_ON_CONFLICT);
    }
    catch(...)
    {
    }
}


void MainWindowController::on_add_camera()
{
    m_gui.server_post(
        boost::bind(&MainWindowController::add_camera_to_scene, this, _1));
}


void MainWindowController::add_camera_to_scene(ServerData &sd)
{
    SceneObject *root = sd.m_scene->get_root();
    CameraObject *new_cam = new CameraObject();
    root->attach(new_cam, peekabot::AUTO_ENUMERATE_ON_CONFLICT);
}


// NOTE: EXECUTED IN THE SERVER
void MainWindowController::load_scene(ServerData &sd, std::string filename)
{
    Path path(
        filename,
        boost::shared_ptr<UploadCache>(),
        m_gui.get_config().get<std::list<boost::filesystem::path> >("data_paths"));

    try
    {
        std::vector<SceneObject *> objs = SceneFileLoader::load_scene(
            m_gui.get_config(), path);

        SceneObject *root = sd.m_scene->get_root();
        BOOST_FOREACH( SceneObject *p, objs )
        {
            root->attach(p, AUTO_ENUMERATE_ON_CONFLICT);
        }

        m_gui.post(boost::bind(
                 &MainWindowController::set_status_message,
                 this,
                 "Scene file " + path.get_expanded_path().string() + " loaded"));
    }
    catch(std::exception &e)
    {
        std::stringstream ss;
        ss << "Failed to load scene file: " << e.what();
        m_gui.post(boost::bind(
                 &MainWindowController::display_message,
                 this, Gtk::MESSAGE_ERROR, ss.str()));
    }
    catch(...)
    {
        m_gui.post(boost::bind(
                 &MainWindowController::display_message,
                 this, Gtk::MESSAGE_ERROR,
                 "Unknown exception caught when loading scene file"));
    }
}


void MainWindowController::on_fullscreen()
{
    if( m_fullscreen_action->get_active() )
    {
        m_main_window->fullscreen();
    }
    else
    {
        m_main_window->unfullscreen();
    }
}


void MainWindowController::on_toggle_sidebar()
{
    m_sidebar_alignment->set_visible(!m_sidebar_alignment->get_visible());
}


void MainWindowController::on_toggle_toolbar()
{
    m_toolbar->set_visible(!m_toolbar->get_visible());
}


void MainWindowController::on_toggle_statusbar()
{
    m_statusbar->set_visible(!m_statusbar->get_visible());
}


void MainWindowController::on_quit()
{
    m_main_window->hide();
}


void MainWindowController::on_infobar_response(int response)
{
    m_infobar.hide();
}


void MainWindowController::display_message(Gtk::MessageType type, const std::string &msg)
{
    if( type == Gtk::MESSAGE_ERROR )
    {
        m_infobar_icon.set(Gtk::Stock::DIALOG_ERROR, Gtk::ICON_SIZE_DIALOG);
        m_infobar_icon.show();
    }
    else if( type == Gtk::MESSAGE_WARNING )
    {
        m_infobar_icon.set(Gtk::Stock::DIALOG_WARNING, Gtk::ICON_SIZE_DIALOG);
        m_infobar_icon.show();
    }
    else if( type == Gtk::MESSAGE_INFO )
    {
        m_infobar_icon.set(Gtk::Stock::DIALOG_INFO, Gtk::ICON_SIZE_DIALOG);
        m_infobar_icon.show();
    }
    else
    {
        m_infobar_icon.hide();
    }

    //m_infobar.set_message_type(type);
    m_infobar_lbl.set_text(msg);
    m_infobar.show();
}


void MainWindowController::show_screenshot_dialog(ObjectID cam_id, const bool *layers)
{
    // Set the initial state of the layer buttons based on the passed
    // layers parameter
    typedef Gtk::Table::TableList::iterator Iter;
    Gtk::Table::TableList &tl = m_screenshot_layer_table->children();
    int c = 10;

    for( Iter it = tl.begin(); it != tl.end(); ++it )
    {
        Gtk::ToggleButton *tb =
            dynamic_cast<Gtk::ToggleButton *>((*it).get_widget());
        assert( tb );
        tb->set_active(layers[--c]);
    }

    m_screenshot_filename_entry->set_text(
        (m_gui.get_config().get_option<boost::filesystem::path>(
            "resources.snapshot_path",
            fs::get_resource_path() / "snapshots") / "screenshot.png"
            ).string());

    m_screenshot_dialog->show();
}


void MainWindowController::on_screenshot_dialog_response(int response)
{
    m_screenshot_dialog->hide();
    if( response == Gtk::RESPONSE_OK )
    {
        bool layers[10];

        // Retrieve the set of selected layers
        typedef Gtk::Table::TableList::iterator Iter;
        Gtk::Table::TableList &tl = m_screenshot_layer_table->children();
        int c = 10;

        for( Iter it = tl.begin(); it != tl.end(); ++it )
        {
            Gtk::ToggleButton *tb =
                dynamic_cast<Gtk::ToggleButton *>((*it).get_widget());
            assert( tb );
            layers[--c] = tb->get_active();
        }

        // Determine which camera to use for the screenshot
        ObjectID cam_id = 0;
        Gtk::TreeModel::iterator it = m_screenshot_cameras->get_active();
        if( it )
        {
            Gtk::TreeModel::Row row = *it;
            if( row )
                cam_id = row[m_screenshot_camera_cols.m_id];;
        }

        const std::string filename = m_screenshot_filename_entry->get_text();

        m_gui.save_screenshot(
            m_screenshot_w_spinbutton->get_value_as_int(),
            m_screenshot_h_spinbutton->get_value_as_int(),
            cam_id, layers, filename);

        set_status_message("Screenshot saved to " + filename);
    }
}


void MainWindowController::on_screenshot_save_response(int response)
{
    m_screenshot_save_dialog->hide();
    if( response == Gtk::RESPONSE_OK )
    {
        m_screenshot_filename_entry->set_text(
            m_screenshot_save_dialog->get_filename());
    }
}


void MainWindowController::set_status_message(const std::string &msg)
{
    m_statusbar->pop(0);
    m_statusbar->push(msg, 0);

    if( m_clear_statusbar_timeout_conn.connected() )
        m_clear_statusbar_timeout_conn.disconnect();

    m_clear_statusbar_timeout_conn = Glib::signal_timeout().connect(
        sigc::mem_fun(*this, &MainWindowController::clear_status_message), 5000);
}


void MainWindowController::push_status_message(const std::string &msg)
{
    m_statusbar->push(msg, 1);
}


void MainWindowController::pop_status_message()
{
    m_statusbar->pop(1);
}


bool MainWindowController::clear_status_message()
{
    m_statusbar->pop(0);

    // Disconnect after clearing
    return false;
}


void MainWindowController::on_delete()
{
    m_gui.server_post(
        boost::bind(
            &MainWindowController::delete_objects, this,
            _1, m_gui.get_selection()));
}


void MainWindowController::delete_objects(
    ServerData &sd, std::set<ObjectID> objs)
{
    BOOST_FOREACH( ObjectID id, objs )
    {
        SceneObject *obj = sd.m_scene->get_object(id);
        if( obj )
        {
            obj->detach();
            delete obj;
        }
    }
}


void MainWindowController::on_selection_changed()
{
    m_delete_action->set_sensitive(!m_gui.get_selection().empty());
}


void MainWindowController::on_fps_updated(float fps)
{
    std::ostringstream ss;
    ss.precision(1);
    ss.setf(std::ios::fixed);
    ss << fps << " fps";
    m_fps_label->set_text(ss.str());
}

