/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GUI_MAIN_WINDOW_CONTROLLER_HH_INCLUDED
#define PEEKABOT_GUI_MAIN_WINDOW_CONTROLLER_HH_INCLUDED


#include "../Types.hh"

#include <set>
#include <gtkmm.h>
#include <boost/scoped_ptr.hpp>
#include <boost/signals2/connection.hpp>


namespace peekabot
{
    class ServerData;

    namespace gui
    {
        class Gui;
        class PreferencesDialog;

        /**
         * \brief Class responsible for overseeing the interaction of the GUI
         * main window, e.g. handling menus and tool buttons etc.
         *
         * The MainWindowController class takes care of everything not handled
         * by the other, more specialized GUI controller classes such as
         * PropInspectorController.
         */
        class MainWindowController
        {
        public:
            MainWindowController(Gui &gui);

            ~MainWindowController();

            void display_message(Gtk::MessageType type, const std::string &msg);

            void set_status_message(const std::string &msg);

            void push_status_message(const std::string &msg);

            void pop_status_message();

            void show_screenshot_dialog(ObjectID cam_id, const bool *layers);

        private:
            void setup_widgets();

            void setup_actions();

            void setup_infobar();

            void add_camera(
                ObjectID id, ObjectID parent_id,
                std::string name, ObjectType type);

            void remove_camera(ObjectID id);

            void on_about();

            void on_about_response(int);

            void on_load_scene();

            void on_load_scene_response(int response);

            void on_clear_scene();

            void on_add_camera();

            void on_fullscreen();

            void on_toggle_sidebar();

            void on_toggle_toolbar();

            void on_toggle_statusbar();

            void on_quit();

            void on_infobar_response(int response);

            void on_screenshot_dialog_response(int response);

            void on_screenshot_save_response(int response);

            void on_delete();

            bool clear_status_message();

            void on_selection_changed();

            void on_fps_updated(float fps);

            /// \name Methods executed in the server thread
            /// @{

            void load_scene(ServerData &sd, std::string filename);

            void clear_scene(ServerData &sd);

            void add_camera_to_scene(ServerData &sd);

            void delete_objects(ServerData &sd, std::set<ObjectID> objs);

            /// @}

        private:
            Gui &m_gui;

            // Widgets

            Gtk::Window *m_main_window;

            Gtk::HPaned *m_main_paned;

            Gtk::VPaned *m_sidebar_paned;

            Gtk::Alignment *m_sidebar_alignment;

            Gtk::AboutDialog *m_about_dialog;

            Gtk::FileChooserDialog *m_load_scene_dialog;

            Gtk::Toolbar *m_toolbar;

            Gtk::MenuBar *m_menubar;

            Gtk::Statusbar *m_statusbar;

            Gtk::VBox *m_main_area_vbox;

            Gtk::InfoBar m_infobar;

            Gtk::HBox m_infobar_hbox;

            Gtk::Label m_infobar_lbl;

            Gtk::Image m_infobar_icon;

            Gtk::Dialog *m_screenshot_dialog;

            Gtk::SpinButton *m_screenshot_w_spinbutton;
            Gtk::SpinButton *m_screenshot_h_spinbutton;

            Gtk::Table *m_screenshot_layer_table;

            Gtk::Entry *m_screenshot_filename_entry;

            Gtk::ComboBox *m_screenshot_cameras;
            Glib::RefPtr<Gtk::ListStore> m_screenshot_cameras_model;

            Gtk::FileChooserDialog *m_screenshot_save_dialog;

            Gtk::Button *m_screenshot_browse_button;

            class CameraCols : public Gtk::TreeModel::ColumnRecord
            {
            public:
                CameraCols()
                {
                    add(m_id);
                    add(m_name);
                }

                Gtk::TreeModelColumn<ObjectID> m_id;
                Gtk::TreeModelColumn<std::string> m_name;
            };

            CameraCols m_screenshot_camera_cols;

            Gtk::Label *m_fps_label;

            // Actions

            Glib::RefPtr<Gtk::Action> m_about_action;

            Glib::RefPtr<Gtk::Action> m_quit_action;

            Glib::RefPtr<Gtk::Action> m_load_scene_action;

            Glib::RefPtr<Gtk::Action> m_clear_scene_action;

            Glib::RefPtr<Gtk::Action> m_add_camera_action;

            Glib::RefPtr<Gtk::ToggleAction> m_fullscreen_action;

            Glib::RefPtr<Gtk::ToggleAction> m_sidebar_action;

            Glib::RefPtr<Gtk::ToggleAction> m_toggle_toolbar_action;

            Glib::RefPtr<Gtk::ToggleAction> m_toggle_statusbar_action;

            Glib::RefPtr<Gtk::Action> m_delete_action;


            boost::scoped_ptr<PreferencesDialog> m_preferences_dialog;

            sigc::connection m_clear_statusbar_timeout_conn;

            boost::signals2::scoped_connection m_fps_updated_conn;
        };
    }
}


#endif // PEEKABOT_GUI_MAIN_WINDOW_CONTROLLER_HH_INCLUDED
