/*
 * Copyright Staffan Gimåker 2008-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MoveTool.hh"
#include "Gui.hh"
#include "ToolController.hh"
#include "SceneViewFrame.hh"
#include "RendererRelay.hh"
#include "MainWindowController.hh"
#include "../Server.hh"
#include "../CameraObject.hh"
#include "../SceneTree.hh"

#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <Eigen/Geometry>
#include <Eigen/LU>


using namespace peekabot;
using namespace peekabot::gui;


namespace
{
    CoordinateSystem idx_to_csys[] = {
        CAMERA_COORDINATES,
        WORLD_COORDINATES,
        PARENT_COORDINATES,
        LOCAL_COORDINATES
    };
}


MoveTool::MoveTool(Gui &gui)
    : m_gui(gui)
{
    m_gui.get_builder()->get_widget(
        "move_options_viewport", m_options_viewport);

    m_gui.get_builder()->get_widget("move_options_table", m_table);

    // Set up the axis buttons
    m_gui.get_builder()->get_widget("move_x_axis_button", m_axis_button[0]);
    m_gui.get_builder()->get_widget("move_y_axis_button", m_axis_button[1]);
    m_gui.get_builder()->get_widget("move_z_axis_button", m_axis_button[2]);
    for( int i = 0; i < 3; ++i )
        m_axis_button[i]->signal_toggled().connect(
            sigc::mem_fun(*this, &MoveTool::on_axis_toggled));

    // Set up the coordinate system combo box
    m_system_combobox.append_text("Camera");
    m_system_combobox.append_text("World");
    m_system_combobox.append_text("Parent");
    m_system_combobox.append_text("Local");
    m_system_combobox.signal_changed().connect(
        sigc::mem_fun(*this, &MoveTool::on_system_changed));
    m_table->attach(
        m_system_combobox, 1, 2, 2, 3,
        Gtk::FILL | Gtk::EXPAND, Gtk::FILL);
    m_system_combobox.show();
}


void MoveTool::activate()
{
    m_gui.get_renderer()->set_tg_mode(renderer::MOVE_TRANSFORMATION_GUIDES);

    m_gui.get_main_window_controller().push_status_message(
        "Move tool: g/m toggles the coordinate system, x, y and z "
        "toggles movement axes");

    m_options_viewport->set_visible(true);

    m_system_combobox.set_active(0);
    on_axis_toggled();
    on_system_changed();
}


void MoveTool::deactivate()
{
    m_gui.get_renderer()->set_tg_mode(renderer::NO_TRANSFORMATION_GUIDES);

    m_gui.get_main_window_controller().pop_status_message();

    m_options_viewport->set_visible(false);
}


void MoveTool::button_release(
    SceneViewFrame *sf, const GdkEventButton *event)
{
    if( event->button == 3 )
    {
        // Right "click" -- cancel move
        m_gui.get_tool_controller().activate_nav_tool();
    }
}


void MoveTool::mouse_drag(
    SceneViewFrame *sf, const GdkEventMotion *event,
    double dx, double dy)
{
    if( !(event->state & GDK_BUTTON1_MASK) )
        return;

    Axis axes = Axis(0);
    if( m_axis_button[0]->get_active() )
        axes = (Axis)(axes | X_AXIS);
    if( m_axis_button[1]->get_active() )
        axes = (Axis)(axes | Y_AXIS);
    if( m_axis_button[2]->get_active() )
        axes = (Axis)(axes | Z_AXIS);

    if( axes == Axis(0) )
        return;

    double rate_mod = 1.0/50;
    if( event->state & GDK_CONTROL_MASK )
        rate_mod /= 5;
    else if( event->state & GDK_SHIFT_MASK )
        rate_mod *= 5;

    m_gui.server_post(
        boost::bind(&MoveTool::move, this, _1,
                    m_gui.get_selection(), sf->get_camera(),
                    idx_to_csys[m_system_combobox.get_active_row_number()],
                    axes, dx*rate_mod, dy*rate_mod));
}


void MoveTool::key_release(
    SceneViewFrame *sf, const GdkEventKey *event)
{
    if( event->keyval == GDK_Escape )
    {
        m_gui.get_tool_controller().activate_nav_tool();
    }
    else if( event->keyval == GDK_r )
    {
        m_gui.get_tool_controller().activate_rotate_tool();
    }
    else if( event->keyval == GDK_g || event->keyval == GDK_m )
    {
        // Toggle coordinate system
        m_system_combobox.set_active(
            (m_system_combobox.get_active_row_number()+1)%4);
    }
    else if( event->keyval == GDK_x )
    {
        if( m_axis_button[0]->is_sensitive() )
            m_axis_button[0]->set_active(!m_axis_button[0]->get_active());
    }
    else if( event->keyval == GDK_y )
    {
        m_axis_button[1]->set_active(!m_axis_button[1]->get_active());
    }
    else if( event->keyval == GDK_z )
    {
        m_axis_button[2]->set_active(!m_axis_button[2]->get_active());
    }
}


void MoveTool::on_axis_toggled()
{
    Axis axes = Axis(0);
    if( m_axis_button[0]->get_active() )
        axes = (Axis)(axes | X_AXIS);
    if( m_axis_button[1]->get_active() )
        axes = (Axis)(axes | Y_AXIS);
    if( m_axis_button[2]->get_active() )
        axes = (Axis)(axes | Z_AXIS);

    m_gui.get_renderer()->set_tg_axes(axes);
}


void MoveTool::on_system_changed()
{
    m_gui.get_renderer()->set_tg_coord_sys(
        idx_to_csys[m_system_combobox.get_active_row_number()]);

    if( m_system_combobox.get_active_row_number() == 0 )
    {
        m_axis_button[0]->set_sensitive(false);
        m_axis_button[0]->set_active(false);
        m_axis_button[1]->set_active(true);
        m_axis_button[2]->set_active(true);
    }
    else
    {
        m_axis_button[0]->set_sensitive(true);
        m_axis_button[0]->set_active(true);
        m_axis_button[1]->set_active(true);
        m_axis_button[2]->set_active(false);
    }
}


void MoveTool::move(
    ServerData &sd, std::set<ObjectID> selection,
    ObjectID cam_id, CoordinateSystem system, Axis axes,
    double dx, double dy)
{
    const CameraObject *cam = dynamic_cast<const CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( !cam )
        return;

    //
    // The code below was adapted from ViewportMove.cc -- how it works
    // is beyond me /Staffan
    //

    Eigen::Transform3f camera_mtow = cam->get_mtow();
    Eigen::Transform3f cam_inverse;
    cam_inverse = Eigen::Transform3f(
        cam->get_mtow().inverse(Eigen::Isometry));

    BOOST_FOREACH( ObjectID id, selection )
    {
        SceneObject *obj = sd.m_scene->get_object(id);

        if( !obj )
            continue;

        // Then, in order to do a mtow->mtop translation, we need
        // the mtow of the target object's parent...
        Eigen::Transform3f parent_mtow = obj->get_parent_mtow();

        Eigen::Transform3f mtow = obj->get_mtow();

        Eigen::Transform3f trans_matrix = obj->get_transformation();

        // We currently support only world coordinates or local coordinates.
        if(system == PARENT_COORDINATES)
        {
            Eigen::Vector4f trans(0, -dx, -dy, 1);
            parent_mtow.matrix().col(3) = Eigen::Vector4f(0, 0, 0, 1);
            camera_mtow.matrix().col(3) = Eigen::Vector4f(0, 0, 0, 1);
            trans = Eigen::Transform3f(
                parent_mtow.inverse(Eigen::Isometry)) *
                camera_mtow *
                trans;
            trans_matrix(0, 3) += ((axes & X_AXIS) > 0) * trans(0);
            trans_matrix(1, 3) += ((axes & Y_AXIS) > 0) * trans(1);
            trans_matrix(2, 3) += ((axes & Z_AXIS) > 0) * trans(2);
        }
        else if(system == WORLD_COORDINATES)
        {
            Eigen::Vector4f trans(0, -dx, -dy , 1);
            Eigen::Transform3f par_mtow_inv = Eigen::Transform3f(
                parent_mtow.inverse(Eigen::Isometry));
            camera_mtow.matrix().col(3) = Eigen::Vector4f(0, 0, 0, 1);
            trans = camera_mtow * trans;
            trans_matrix = (par_mtow_inv *
                            Eigen::Translation3f(
                                ((axes & X_AXIS) > 0) * trans(0),
                                ((axes & Y_AXIS) > 0) * trans(1),
                                ((axes & Z_AXIS) > 0) * trans(2))
                            * mtow);
        }

        else if(system == LOCAL_COORDINATES)
        {
            Eigen::Vector4f trans(0, -dx, -dy , 1);
            mtow.matrix().col(3) = Eigen::Vector4f(0, 0, 0, 1);
            camera_mtow.matrix().col(3) = Eigen::Vector4f(0, 0, 0, 1);
            trans = Eigen::Transform3f(
                mtow.inverse(Eigen::Isometry)) * camera_mtow * trans;
            trans_matrix.matrix().col(3) += (
                (axes & X_AXIS) > 0) *
                trans(0) *
                trans_matrix.matrix().col(0);
            trans_matrix.matrix().col(3) += (
                (axes & Y_AXIS) > 0) *
                trans(1) *
                trans_matrix.matrix().col(1);
            trans_matrix.matrix().col(3) += (
                (axes & Z_AXIS) > 0) *
                trans(2) *
                trans_matrix.matrix().col(2);
        }
        else if(system == CAMERA_COORDINATES)
        {
            // Apply transformation
            mtow.matrix().col(3) += (
                ((axes & Y_AXIS) > 0) * dx * -camera_mtow.matrix().col(1) +
                ((axes & Z_AXIS) > 0) * dy * -camera_mtow.matrix().col(2));

            trans_matrix = Eigen::Transform3f(
                parent_mtow.inverse(Eigen::Isometry)) * mtow;
        }

        obj->set_transformation(trans_matrix);
    }
}
