/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MtowPropMediator.hh"
#include "../SceneObject.hh"
#include "../props/MtowPropAdapter.hh"
#include "../Server.hh"
#include "../SceneTree.hh"

#include <cassert>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


MtowPropMediator::MtowPropMediator(
    PropInspectorController &pi, SceneObject *obj,
    MtowPropAdapter *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0)
{
    m_mtow = m_obj->get_mtow();
    m_mtop = m_obj->get_transformation();

    // Create and init widget
    post(
        boost::bind(&MtowPropMediator::create_widgets, this,
                    m_obj->get_prop_name(m_prop_key)));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&MtowPropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void MtowPropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(
        boost::bind(&MtowPropMediator::destroy_widgets, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&MtowPropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void MtowPropMediator::create_widgets(std::string prop_name)
{
    for( int i = 0; i < 3; ++i )
    {
        m_xyz[i] = new Gtk::SpinButton();
        m_rpy[i] = new Gtk::SpinButton();

        m_xyz[i]->set_numeric(true);
        m_rpy[i]->set_numeric(true);
        m_xyz[i]->set_range(
            -std::numeric_limits<float>::max(),
            std::numeric_limits<float>::max());
        m_xyz[i]->set_increments(0.1, 1);
        m_rpy[i]->set_increments(1, 10);
        m_xyz[i]->set_width_chars(3);
        m_rpy[i]->set_width_chars(3);
        m_xyz[i]->set_digits(2);
        m_rpy[i]->set_digits(2);

        m_xyz_set_conn[i] = m_xyz[i]->signal_value_changed().connect(
            sigc::mem_fun(*this, &MtowPropMediator::on_xyz_set));

        m_rpy_set_conn[i] = m_rpy[i]->signal_value_changed().connect(
            sigc::mem_fun(*this, &MtowPropMediator::on_rpy_set));
    }

    m_rpy[0]->set_range(-180, 180);
    m_rpy[1]->set_range(-90, 90);
    m_rpy[2]->set_range(-180, 180);

    m_csys_box = new Gtk::ComboBoxText();
    m_csys_box->append_text("World coordinates");
    m_csys_box->append_text("Parent coordinates");
    m_csys_box->set_active(0);
    m_world_coords = true;
    m_csys_box->signal_changed().connect(
        sigc::mem_fun(*this, &MtowPropMediator::on_csys_changed));

    m_rpy_lbl = new Gtk::Label("RPY:", 1.0, 0.5);
    m_rpy_lbl->set_tooltip_text(
        "The rotation in X-Y-Z Tait-Bryan angles. "
        "All rotations are extrinsic, i.e. a fixed coordinate system is "
        "used for rotational axes. The Rotations are performed in the order "
        "X (roll), then Y (pitch) and Z (yaw) last.");
    m_xyz_lbl = new Gtk::Label("Position:", 1.0, 0.5);
    m_xyz_hbox = new Gtk::HBox();
    m_rpy_hbox = new Gtk::HBox();
    m_xyz_hbox->set_spacing(3);
    m_rpy_hbox->set_spacing(3);

    for( int i = 0; i < 3; ++i )
    {
        m_xyz_hbox->pack_start(*m_xyz[i], true, true);
        m_rpy_hbox->pack_start(*m_rpy[i], true, true);
    }

    update_widgets();

    add_prop_widgets(m_xyz_lbl, m_xyz_hbox);
    add_prop_widgets(m_rpy_lbl, m_rpy_hbox);
    add_prop_widgets(0, m_csys_box);
}


// Executed in the GUI thread
void MtowPropMediator::destroy_widgets()
{
    erase_prop_widgets(m_xyz_lbl, m_xyz_hbox);
    erase_prop_widgets(m_rpy_lbl, m_rpy_hbox);
    erase_prop_widgets(0, m_csys_box);

    for( int i = 0; i < 3; ++i )
    {
        delete m_xyz[i];
        m_xyz[i] = 0;
        delete m_rpy[i];
        m_rpy[i] = 0;
    }

    delete m_xyz_hbox;
    m_xyz_hbox = 0;
    delete m_rpy_hbox;
    m_rpy_hbox = 0;
    delete m_csys_box;
    m_csys_box = 0;
    delete m_xyz_lbl;
    m_xyz_lbl = 0;
    delete m_rpy_lbl;
    m_rpy_lbl = 0;

    delete this;
}


// Executed in the server thread
void MtowPropMediator::on_prop_set()
{
    post(
        boost::bind(
            &MtowPropMediator::update_values, this,
            m_obj->get_mtow(), m_obj->get_transformation()));
}


// Executed in the GUI thread
void MtowPropMediator::update_values(
    Eigen::Transform3f mtow, Eigen::Transform3f mtop)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    m_mtow = mtow;
    m_mtop = mtop;

    update_widgets();
}


void MtowPropMediator::update_widgets()
{
    const Eigen::Transform3f &m = m_world_coords ? m_mtow : m_mtop;

    float theta, phi, psi;
    float theta1, phi1, psi1;
    float theta2, phi2, psi2;

    if( fabsf(m.linear()(2,0)) != 1 )
    {
        theta1 = -asin(m.linear()(2,0));
        theta2 = M_PI - theta1;
        float costheta1 = cosf(theta1);
        float costheta2 = cosf(theta2);
        psi1 = atan2(m.linear()(2,1)/costheta1, m.linear()(2,2)/costheta1);
        phi1 = atan2(m.linear()(1,0)/costheta1, m.linear()(0,0)/costheta1);
        psi2 = atan2(m.linear()(2,1)/costheta2, m.linear()(2,2)/costheta2);
        phi2 = atan2(m.linear()(1,0)/costheta2, m.linear()(0,0)/costheta2);

        // When cos(theta) != 0 there are two solutions, we'll use the first
        // for now.
        theta = theta1;
        phi = phi1;
        psi = psi1;
    }
    else
    {
        // phi is unconstrained in this case, so we can set it to anything
        phi = m_rpy[2]->get_value()/180*M_PI;
        if( m.linear()(2,0) == -1 )
        {
            theta = M_PI_2;
            psi = phi + atan2(m.linear()(0,1), m.linear()(0,2));
        }
        else
        {
            theta = -M_PI_2;
            psi = -phi + atan2(-m.linear()(0,1), -m.linear()(0,2));
        }
    }

    float rpy[3] = { psi, theta, phi };

    for( int i = 0; i < 3; ++i )
    {
        m_xyz_set_conn[i].block();
        m_rpy_set_conn[i].block();
        m_xyz[i]->set_value(m.translation()(i));
        m_rpy[i]->set_value(rpy[i]*180/M_PI);
        m_rpy_set_conn[i].unblock();
        m_xyz_set_conn[i].unblock();
    }
}


// Executed in the GUI thread
void MtowPropMediator::on_xyz_set()
{
    ++m_n_updates_queued;
    server_post(
        boost::bind(
            &MtowPropMediator::set_position, _1,
            m_object_id, Eigen::Vector3f(
                m_xyz[0]->get_value(),
                m_xyz[1]->get_value(),
                m_xyz[2]->get_value()), m_world_coords));
}


void MtowPropMediator::on_rpy_set()
{
    ++m_n_updates_queued;

    float yaw = m_rpy[2]->get_value()/180*M_PI;
    float pitch = m_rpy[1]->get_value()/180*M_PI;
    float roll = m_rpy[0]->get_value()/180*M_PI;

    server_post(
        boost::bind(
            &MtowPropMediator::set_rpy, _1,
            m_object_id, roll, pitch, yaw, m_world_coords));
}


void MtowPropMediator::on_csys_changed()
{
    bool world_coords = m_csys_box->get_active_row_number() == 0;
    if( world_coords != m_world_coords )
    {
        m_world_coords = world_coords;
        update_widgets();
    }
}


void MtowPropMediator::set_position(
    ServerData &sd, ObjectID object_id,
    Eigen::Vector3f pos, bool world_coords)
{
    SceneObject *obj = sd.m_scene->get_object(object_id);
    if( obj )
    {
        if( world_coords )
            obj->set_world_position(pos);
        else
            obj->set_position(pos);
    }
}


void MtowPropMediator::set_rpy(
    ServerData &sd, ObjectID object_id,
    float roll, float pitch, float yaw, bool world_coords)
{
    SceneObject *obj = sd.m_scene->get_object(object_id);
    if( obj )
    {
        obj->set_rpy(
            roll, pitch, yaw,
            world_coords ? WORLD_COORDINATES : PARENT_COORDINATES);
    }
}
