/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "PreferencesDialog.hh"
#include "Gui.hh"
#include "../Config.hh"
#include "../FsToolbox.hh"


using namespace peekabot;
using namespace peekabot::gui;


class PreferencesDialog::DataPathCols : public Gtk::TreeModelColumnRecord
{
public:
    DataPathCols()
    {
        add(m_path);
    }

    Gtk::TreeModelColumn<std::string> m_path;
};


PreferencesDialog::PreferencesDialog(Gui &gui)
    : m_gui(gui),
      m_config(gui.get_config()),
      m_dp_cols(0)
{
    m_gui.get_builder()->get_widget("preferences_dialog", m_dialog);
    m_dialog->signal_hide().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_hide));
    m_dialog->signal_show().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_show));

    m_preferences_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        m_gui.get_builder()->get_object("preferences_action"));
    m_preferences_action->signal_activate().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_preferences_activated));

    m_gui.get_builder()->get_widget(
        "preferences_close_button", m_close_button);
    m_close_button->signal_clicked().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_close_clicked));

    m_gui.get_builder()->get_widget(
        "screenshot_dir_button", m_screenshot_dir_button);

    m_gui.get_builder()->get_widget(
        "data_paths_treeview", m_dp_view);
    m_dp_cols = new DataPathCols();
    m_dp_model = Gtk::ListStore::create(*m_dp_cols);
    m_dp_view->set_model(m_dp_model);
    m_dp_view->append_column("Data path", m_dp_cols->m_path);
    m_dp_selection = m_dp_view->get_selection();
    m_dp_selection->signal_changed().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_dp_selection_changed));

    m_gui.get_builder()->get_widget(
        "data_paths_add_button", m_dp_add_button);
    m_dp_add_button->signal_clicked().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_dp_add_clicked));

    m_gui.get_builder()->get_widget(
        "data_paths_remove_button", m_dp_remove_button);
    m_dp_remove_button->set_sensitive(false);
    m_dp_remove_button->signal_clicked().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_dp_remove_clicked));

    m_gui.get_builder()->get_widget(
        "add_data_path_dialog", m_dp_add_dialog);
    m_dp_add_dialog->signal_response().connect(
        sigc::mem_fun(*this, &PreferencesDialog::on_dp_add_response));
}


PreferencesDialog::~PreferencesDialog()
{
    if( m_dp_cols )
    {
        delete m_dp_cols;
        m_dp_cols = 0;
    }
}


void PreferencesDialog::on_preferences_activated()
{
    m_dialog->show();
}


void PreferencesDialog::on_close_clicked()
{
    m_dialog->hide();
}


void PreferencesDialog::on_dp_add_clicked()
{
    m_dp_add_dialog->show();
}


void PreferencesDialog::on_dp_remove_clicked()
{
    Gtk::TreeModel::iterator it = m_dp_selection->get_selected();
    assert( it );
    m_dp_model->erase(*it);
}


void PreferencesDialog::on_dp_add_response(int response)
{
    m_dp_add_dialog->hide();
    if( response == Gtk::RESPONSE_OK )
    {
        DataPaths dp = m_config.get<DataPaths>(
            "resources.data_paths", DataPaths());
        std::string new_dp_dir = m_dp_add_dialog->get_filename();
        dp.push_back(boost::filesystem::path(new_dp_dir));
        m_config.set_persistent("resources.data_paths", dp);

        Gtk::TreeModel::iterator iter = m_dp_model->append();
        Gtk::TreeModel::Row row = *iter;
        row[m_dp_cols->m_path] = new_dp_dir;
    }
}


void PreferencesDialog::on_dp_selection_changed()
{
    m_dp_remove_button->set_sensitive(m_dp_selection->count_selected_rows());
}


void PreferencesDialog::on_hide()
{
    //
    // Set the screenshot dir if it has been changed
    //
    const std::string &screenshot_dir = m_screenshot_dir_button->get_filename();
    boost::filesystem::path screenshot_path(screenshot_dir);

    if( m_config.get<boost::filesystem::path>(
            "resources.snapshot_path",
            fs::get_resource_path() / "snapshots") != screenshot_path )
    {
        m_config.set_persistent(
            "resources.snapshot_path", screenshot_path);
    }

    //
    // Set the data paths if they have been changed
    //
    DataPaths dp;
    for( Gtk::TreeModel::const_iterator it = m_dp_model->children().begin();
         it != m_dp_model->children().end(); ++it )
    {
        const std::string &s = (*it)[m_dp_cols->m_path];
        dp.push_back(boost::filesystem::path(s));
    }

    if( m_config.get<DataPaths>("resources.data_paths", DataPaths()) != dp )
        m_config.set_persistent("resources.data_paths", dp);

    //
    // Save the configuration
    //
    boost::filesystem::path conf_path =
        fs::get_resource_path() / fs::make_path("peekabot.conf");
    m_config.save(conf_path.string());
}


void PreferencesDialog::on_show()
{
    // Update the screenshot dir button with the current value
    m_screenshot_dir_button->set_filename(
        m_config.get<boost::filesystem::path>(
            "resources.snapshot_path",
            fs::get_resource_path() / "snapshots").string());


    // Clear and re-populate the data paths tree view
    while( !m_dp_model->children().empty() )
        m_dp_model->erase(*m_dp_model->children().begin());

    DataPaths dp = m_config.get<DataPaths>("resources.data_paths", DataPaths());
    for( DataPaths::const_iterator it = dp.begin(); it != dp.end(); ++it )
    {
        Gtk::TreeModel::iterator iter = m_dp_model->append();
        Gtk::TreeModel::Row row = *iter;
        row[m_dp_cols->m_path] = it->string();
    }
}
