/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "PropInspectorController.hh"
#include "PropMediator.hh"
#include "PropMediatorFactory.hh"
#include "Gui.hh"
#include "../SceneObject.hh"

#include <cassert>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


PropInspectorController::PropInspectorController(Gui &gui)
    : m_gui(gui),
      m_prop_rows(0)
{
    const Glib::RefPtr<Gtk::Builder> &builder = m_gui.get_builder();
    builder->get_widget("prop_table", m_prop_table);
    builder->get_widget("prop_image", m_prop_img);
    builder->get_widget("prop_label", m_prop_lbl);

    m_gui.ss_selection_changed_signal().connect(
        boost::bind(&PropInspectorController::ss_on_selection_changed, this));
}


PropInspectorController::~PropInspectorController()
{
}


void PropInspectorController::ss_on_selection_changed()
{
    std::set<SceneObject *> sel = m_gui.ss_get_selection();
    if( sel.size() == 1 )
    {
        add_prop_mediators(*sel.begin());
    }
    else
    {
        clear_prop_mediators();
    }
}


// NOTE: EXECUTED IN THE SERVER
void PropInspectorController::add_prop_mediators(SceneObject *obj)
{
    PropMediatorFactory factory(*this);

    for( SceneObject::PropIterator it = obj->prop_adapters_begin();
         it != obj->prop_adapters_end(); ++it )
    {
        PropMediator *mediator = 0;

        try
        {
            mediator = factory.create_mediator(obj, it->first);
            if( mediator )
                m_ss_prop_mediators.insert(mediator);
        }
        catch(...)
        {
            if( mediator )
            {
                mediator->destroy();
                m_ss_prop_mediators.erase(mediator);
            }
        }
    }

    m_gui.post(
        boost::bind(
            &PropInspectorController::show_prop_widgets,
            this, obj->get_object_type()));
}


// NOTE: EXECUTED IN THE SERVER
void PropInspectorController::clear_prop_mediators()
{
    m_gui.post(boost::bind(&PropInspectorController::hide_prop_widgets, this));

    typedef std::set<PropMediator *>::iterator Iter;
    for( Iter it = m_ss_prop_mediators.begin();
         it != m_ss_prop_mediators.end(); ++it )
    {
        (*it)->destroy();
    }
    m_ss_prop_mediators.clear();
}


void PropInspectorController::add_prop_widgets(Gtk::Widget *lbl, Gtk::Widget *w)
{
    if( lbl )
    {
        m_prop_table->attach(
            *lbl, 0, 1, m_prop_rows, m_prop_rows+1,
            Gtk::FILL | Gtk::SHRINK, Gtk::FILL);
        lbl->show_all();
    }

    assert( w );
    m_prop_table->attach(
        *w, 1, 2, m_prop_rows, m_prop_rows+1,
        Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
    w->show_all();

    ++m_prop_rows;
}


void PropInspectorController::erase_prop_widgets(Gtk::Widget *lbl, Gtk::Widget *w)
{
    // TODO

    assert( m_prop_rows > 0 );
    --m_prop_rows;
}


void PropInspectorController::show_prop_widgets(ObjectType type)
{
    m_prop_img->set(m_gui.get_icon(type));
    m_prop_lbl->show();
    m_prop_img->show();
}


void PropInspectorController::hide_prop_widgets()
{
    m_prop_lbl->hide();
    m_prop_img->hide();
}
