/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GUI_PROP_MEDIATOR_HH_INCLUDED
#define PEEKABOT_GUI_PROP_MEDIATOR_HH_INCLUDED


#include "../Types.hh"
#include "../Any.hh"

#include <gtkmm.h>
#include <boost/bind.hpp>
#include <boost/function.hpp>


namespace peekabot
{
    class ServerData;

    namespace gui
    {
        class PropInspectorController;

        /**
         * \brief A PropMediator is responsible for directing the interactions
         * between a property and its graphical representation.
         *
         * This includes creating and destroying the widget, adding and
         * removing it from its parent, updating the widget when the prop
         * changes and updating the prop when a new value is input in the
         * widget.
         */
        class PropMediator
        {
        public:
            /**
             * \brief Schedule the mediator and associated widget for
             * destruction.
             *
             * Calling this method will remove the mediator's widget from the
             * property inspector, disconnect all slots and free any resources
             * used by the mediator object itself.
             *
             * Must be called from within the server thread.
             */
            virtual void destroy() = 0;

        protected:
            PropMediator(PropInspectorController &pi);

            virtual ~PropMediator();

            void post(const boost::function<void ()> &handler);

            void server_post(
                const boost::function<void (ServerData &)> &handler);

            PropInspectorController &get_prop_inspector_controller();

            const PropInspectorController &
            get_prop_inspector_controller() const;

            template<typename T>
            void set_prop(ObjectID object_id, PropKey prop_key, const T &val)
            {
                server_post(
                    boost::bind(
                        &PropMediator::ss_set_prop, this,
                        _1, object_id, prop_key, Any(val)));
            }

            void add_prop_widgets(Gtk::Widget *lbl, Gtk::Widget *widget);

            void erase_prop_widgets(Gtk::Widget *lbl, Gtk::Widget *widget);

        private:
            void ss_set_prop(
                ServerData &sd, ObjectID object_id, PropKey prop_key, Any val);

        private:
            PropInspectorController &m_prop_inspector_controller;
        };
    }
}


#endif // PEEKABOT_GUI_PROP_MEDIATOR_HH_INCLUDED
