/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GUI_RENDERER_RELAY_HH_INCLUDED
#define PEEKABOT_GUI_RENDERER_RELAY_HH_INCLUDED


#include "../Types.hh"
#include "../renderer/Types.hh"

#include <boost/utility.hpp>
#include <boost/signals2.hpp>


namespace peekabot
{
    namespace gui
    {
        class Gui;

        /**
         * \brief Class responsible for interfacing the rendering engine,
         * and to make sure the contents in the renderer are synchronized with
         * the main scene tree.
         *
         * This class is a GUI component and as such all methods (including
         * creation) must be done in the GUI thread. It's not safe to call any
         * of its methods from outside the GUI thread.
         *
         * Instances of this class does not configure themselves according to
         * the configuration file. Max FPS, background color, etc. should be
         * configured by the user (of the instance). Max FPS etc. are set to
         * undefined (but valid) values when instantiated.
         */
        class RendererRelay : public boost::noncopyable
        {
        public:
            RendererRelay(Gui &gui);

            ~RendererRelay();

            void render(
                unsigned int w, unsigned int h, ObjectID camera_id,
                const bool *visible_layers);

            ObjectID pick(
                unsigned int w, unsigned int h, ObjectID camera_id,
                const bool *visible_layers,
                int x, int y, float sensitivity);

            void init();


            void set_tg_mode(renderer::TransformationGuidesMode mode);

            void set_tg_coord_sys(CoordinateSystem coord_sys);

            void set_tg_pivot(RotationCenterpoint pivot);

            void set_tg_axes(Axis axes);


            void set_max_fps(float fps);

            void set_background_color(const RGBColor &color);

            void set_selected_color(const RGBColor &color);

            void set_ancestor_selected_color(const RGBColor &color);

            typedef boost::signals2::signal<void (float)> FpsUpdatedSignal;

            FpsUpdatedSignal &fps_updated_signal() const;

        private:
            class Impl;

            Impl *m_impl;
        };
    }
}


#endif // PEEKABOT_GUI_RENDERER_RELAY_HH_INCLUDED
