/*
 * Copyright Staffan Gimåker 2008-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "RotateTool.hh"
#include "Gui.hh"
#include "ToolController.hh"
#include "SceneViewFrame.hh"
#include "RendererRelay.hh"
#include "MainWindowController.hh"
#include "../Server.hh"
#include "../CameraObject.hh"
#include "../SceneTree.hh"

#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <Eigen/Geometry>
#include <Eigen/LU>


using namespace peekabot;
using namespace peekabot::gui;


namespace
{
    CoordinateSystem idx_to_csys[] = {
        CAMERA_COORDINATES,
        WORLD_COORDINATES,
        PARENT_COORDINATES,
        LOCAL_COORDINATES
    };

    RotationCenterpoint idx_to_pivot[] = {
        AVERAGE_CENTER,
        LOCAL_CENTER,
        PARENT_CENTER,
        WORLD_CENTER
    };

    inline float sgnf(float x)
    {
        return (x < 0) ? -1 : 1;
    }
}


RotateTool::RotateTool(Gui &gui)
    : m_gui(gui)
{
    m_gui.get_builder()->get_widget(
        "rotate_options_viewport", m_options_viewport);

    m_gui.get_builder()->get_widget("rotate_options_table", m_table);

    // Set up the axis buttons
    m_gui.get_builder()->get_widget("rotate_x_axis_button", m_axis_button[0]);
    m_gui.get_builder()->get_widget("rotate_y_axis_button", m_axis_button[1]);
    m_gui.get_builder()->get_widget("rotate_z_axis_button", m_axis_button[2]);
    for( int i = 0; i < 3; ++i )
        m_axis_button[i]->signal_toggled().connect(
            sigc::mem_fun(*this, &RotateTool::on_axis_toggled));

    // Set up the coordinate system combo box
    m_system_combobox.append_text("Camera");
    m_system_combobox.append_text("World");
    m_system_combobox.append_text("Parent");
    m_system_combobox.append_text("Local");
    m_system_combobox.signal_changed().connect(
        sigc::mem_fun(*this, &RotateTool::on_system_changed));
    m_table->attach(
        m_system_combobox, 1, 2, 2, 3,
        Gtk::FILL | Gtk::EXPAND, Gtk::FILL);
    m_system_combobox.show();

    // Set up the pivot combo box
    m_pivot_combobox.append_text("Average");
    m_pivot_combobox.append_text("Local");
    m_pivot_combobox.append_text("Parent");
    m_pivot_combobox.append_text("World");
    m_pivot_combobox.signal_changed().connect(
        sigc::mem_fun(*this, &RotateTool::on_pivot_changed));
    m_table->attach(
        m_pivot_combobox, 1, 2, 3, 4,
        Gtk::FILL | Gtk::EXPAND, Gtk::FILL);
    m_pivot_combobox.show();
}


void RotateTool::activate()
{
    m_gui.get_renderer()->set_tg_mode(renderer::ROTATE_TRANSFORMATION_GUIDES);

    m_gui.get_main_window_controller().push_status_message(
        "Rotate tool: r toggles the coordinate system, x, y and z "
        "toggles movement axes");

    m_options_viewport->set_visible(true);

    m_axis_button[0]->set_active(true);
    m_system_combobox.set_active(0);
    m_pivot_combobox.set_active(0);
    on_axis_toggled();
    on_pivot_changed();
    on_system_changed();
}


void RotateTool::deactivate()
{
    m_gui.get_renderer()->set_tg_mode(renderer::NO_TRANSFORMATION_GUIDES);

    m_gui.get_main_window_controller().pop_status_message();

    m_options_viewport->set_visible(false);
}


void RotateTool::button_release(
    SceneViewFrame *sf, const GdkEventButton *event)
{
    if( event->button == 3 )
    {
        // Right "click" -- cancel move
        m_gui.get_tool_controller().activate_nav_tool();
    }
}


void RotateTool::mouse_drag(
    SceneViewFrame *sf, const GdkEventMotion *event,
    double dx, double dy)
{
    if( !(event->state & GDK_BUTTON1_MASK) )
        return;

    Axis axis;
    if( m_axis_button[0]->get_active() )
        axis = X_AXIS;
    else if( m_axis_button[1]->get_active() )
        axis =Y_AXIS;
    else
        axis = Z_AXIS;

    double rate_mod = M_PI / 1800 / 3;
    if( event->state & GDK_CONTROL_MASK )
        rate_mod /= 5;
    else if( event->state & GDK_SHIFT_MASK )
        rate_mod *= 5;

    m_gui.server_post(
        boost::bind(&RotateTool::rotate, this, _1,
                    m_gui.get_selection(), sf->get_camera(),
                    idx_to_csys[m_system_combobox.get_active_row_number()],
                    idx_to_pivot[m_pivot_combobox.get_active_row_number()],
                    axis, dx*rate_mod, dy*rate_mod));
}


void RotateTool::key_release(
    SceneViewFrame *sf, const GdkEventKey *event)
{
    if( event->keyval == GDK_Escape )
    {
        m_gui.get_tool_controller().activate_nav_tool();
    }
    else if( event->keyval == GDK_g || event->keyval == GDK_m )
    {
        m_gui.get_tool_controller().activate_move_tool();
    }
    else if( event->keyval == GDK_r )
    {
        // Toggle coordinate system
        m_system_combobox.set_active(
            (m_system_combobox.get_active_row_number()+1)%4);
    }
    else if( event->keyval == GDK_x )
    {
        m_axis_button[0]->set_active(true);
    }
    else if( event->keyval == GDK_y )
    {
        m_axis_button[1]->set_active(true);
    }
    else if( event->keyval == GDK_z )
    {
        m_axis_button[2]->set_active(true);
    }
}


void RotateTool::on_axis_toggled()
{
    if( m_axis_button[0]->get_active() )
        m_gui.get_renderer()->set_tg_axes(X_AXIS);
    else if( m_axis_button[1]->get_active() )
        m_gui.get_renderer()->set_tg_axes(Y_AXIS);
    else if( m_axis_button[2]->get_active() )
        m_gui.get_renderer()->set_tg_axes(Z_AXIS);
}


void RotateTool::on_system_changed()
{
    if( m_system_combobox.get_active_row_number() == 0 )
        m_gui.get_renderer()->set_tg_coord_sys(CAMERA_COORDINATES);
    else if( m_system_combobox.get_active_row_number() == 1 )
        m_gui.get_renderer()->set_tg_coord_sys(WORLD_COORDINATES);
    else if( m_system_combobox.get_active_row_number() == 2 )
        m_gui.get_renderer()->set_tg_coord_sys(PARENT_COORDINATES);
    else
        m_gui.get_renderer()->set_tg_coord_sys(LOCAL_COORDINATES);
}


void RotateTool::on_pivot_changed()
{
    if( m_pivot_combobox.get_active_row_number() == 0 )
        m_gui.get_renderer()->set_tg_pivot(AVERAGE_CENTER);
    else if( m_pivot_combobox.get_active_row_number() == 1 )
        m_gui.get_renderer()->set_tg_pivot(LOCAL_CENTER);
    else if( m_pivot_combobox.get_active_row_number() == 2 )
        m_gui.get_renderer()->set_tg_pivot(PARENT_CENTER);
    else
        m_gui.get_renderer()->set_tg_pivot(WORLD_CENTER);
}


void RotateTool::rotate(
    ServerData &sd, std::set<ObjectID> selection,
    ObjectID cam_id, CoordinateSystem system,
    RotationCenterpoint pivot, Axis axes,
    double dx, double dy)
{
    const CameraObject *cam = dynamic_cast<const CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( !cam )
        return;

    //
    // The code below was adapted from ViewportRotate.cc -- how it works
    // is beyond me /Staffan
    //

    // First, calculate the average centerpoint if used
    Eigen::Vector3f average_center(0, 0, 0);
    if(pivot == AVERAGE_CENTER)
    {
        std::size_t n = 0;
        BOOST_FOREACH( ObjectID id, selection )
        {
            SceneObject *obj = sd.m_scene->get_object(id);
            if( obj )
            {
                ++n;
                average_center += obj->get_mtow().translation();
            }
        }

        if(n > 0)
            average_center /= n;
    }

    Eigen::Transform3f cam_mtow = cam->get_mtow();

    // Next, apply transformation to all objects in the set.
    BOOST_FOREACH( ObjectID id, selection )
    {
        SceneObject *obj = sd.m_scene->get_object(id);
        if( !obj )
            continue;

        // First fetch all relevant transformation matrices.
        // We want to transform them into world coordinates
        // so that they are all handled uniformly.

        Eigen::Transform3f trans_matrix = obj->get_mtow();
        Eigen::Transform3f parent_mtow = obj->get_parent_mtow();

        Eigen::Transform3f rotational_matrix;
        float x_amount, y_amount, z_amount;

        if(system == WORLD_COORDINATES)
        {
            rotational_matrix.setIdentity();
        }
        else if(system == LOCAL_COORDINATES)
        {
            rotational_matrix = obj->get_mtow();
        }
        else if(system == PARENT_COORDINATES)
        {
            rotational_matrix = parent_mtow;
        }
        else
        {
            rotational_matrix = cam_mtow;
        }

        // The amount of rotation is scaled by the amount the cursor
        // has moved in the othogonal direction of the rotational axis.

        float yscalar, zscalar;
        yscalar = cam_mtow.linear().col(1).dot(
            rotational_matrix.linear().col(0));
        zscalar = cam_mtow.linear().col(2).dot(
            rotational_matrix.linear().col(0));
        x_amount = ((dx * sgnf(yscalar) * (1 - fabs(yscalar)) +
                     dy * sgnf(zscalar) * (1 - fabs(zscalar))) *
                    (axes & X_AXIS));
        yscalar = cam_mtow.linear().col(1).dot(
            rotational_matrix.linear().col(1));
        zscalar = cam_mtow.linear().col(2).dot(
            rotational_matrix.linear().col(1));
        y_amount = ((dx * sgnf(yscalar) * (1 - fabs(yscalar)) +
                     dy * sgnf(yscalar) * (1 - fabs(zscalar))) *
                    (axes & Y_AXIS));
        yscalar = cam_mtow.linear().col(1).dot(
            rotational_matrix.linear().col(2));
        zscalar = cam_mtow.linear().col(2).dot(
            rotational_matrix.linear().col(2));
        z_amount = ((dx * sgnf(yscalar) * (1 - fabs(yscalar)) +
                     dy * sgnf(yscalar) * (1 - fabs(zscalar))) *
                    (axes & Z_AXIS));

        // Transform pivot coordinates into world
        Eigen::Vector3f tmp(0, 0, 0);
        if(pivot == LOCAL_CENTER)
            tmp = trans_matrix.translation();
        else if(pivot == PARENT_CENTER)
            tmp = parent_mtow.translation();
        else if(pivot == AVERAGE_CENTER)
            tmp = average_center;

        // If WORLD_COORDINATES we don't need to do anything

        trans_matrix.translation() -= tmp;
        trans_matrix = Eigen::AngleAxisf(
            x_amount, rotational_matrix.linear().col(0)) * trans_matrix;
        trans_matrix = Eigen::AngleAxisf(
            y_amount, rotational_matrix.linear().col(1)) * trans_matrix;
        trans_matrix = Eigen::AngleAxisf(
            z_amount, rotational_matrix.linear().col(2)) * trans_matrix;
        trans_matrix.translation() += tmp;

        trans_matrix =
            parent_mtow.inverse(Eigen::Isometry) *
            trans_matrix;

        obj->set_transformation(trans_matrix);
    }
}
