/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "StringPropMediator.hh"
#include "Gui.hh"
#include "../SceneObject.hh"
#include "../props/StringPropBase.hh"

#include <cassert>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


StringPropMediator::StringPropMediator(
    PropInspectorController &pi, SceneObject *obj,
    StringPropBase *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0),
      m_widget(0), m_lbl(0)
{
    // Create and init widget
    post(
        boost::bind(&StringPropMediator::create_widget, this,
                    m_obj->get_prop_name(m_prop_key),
                    any_cast<std::string>(m_prop->get(m_obj)),
                    m_prop->is_read_only()));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&StringPropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void StringPropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(
        boost::bind(&StringPropMediator::destroy_widget, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&StringPropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void StringPropMediator::create_widget(
    std::string prop_name, std::string val, bool read_only)
{
    assert( !m_widget );
    assert( !m_lbl );

    m_widget = new Gtk::Entry();
    m_widget->set_text(val);
    m_widget->set_width_chars(10);
    m_widget_set_conn = m_widget->signal_changed().connect(
        sigc::mem_fun(*this, &StringPropMediator::on_widget_set));
    if( read_only )
        m_widget->set_editable(false);

    m_lbl = new Gtk::Label(prop_name + ":", 1.0, 0.5);

    add_prop_widgets(m_lbl, m_widget);
}


// Executed in the GUI thread
void StringPropMediator::destroy_widget()
{
    assert( m_widget );
    assert( m_lbl );

    erase_prop_widgets(m_lbl, m_widget);
    delete m_widget;
    m_widget = 0;
    delete m_lbl;
    m_lbl = 0;

    delete this;
}


// Executed in the server thread
void StringPropMediator::on_prop_set()
{
    std::string val = any_cast<std::string>(m_prop->get(m_obj));
    post(
        boost::bind(
            &StringPropMediator::update_widget, this, val));
}


// Executed in the GUI thread
void StringPropMediator::update_widget(std::string val)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    m_widget_set_conn.block();
    m_widget->set_text(val);
    m_widget_set_conn.unblock();
}


// Executed in the GUI thread
void StringPropMediator::on_widget_set()
{
    ++m_n_updates_queued;
    set_prop(m_object_id, m_prop_key, std::string(m_widget->get_text()));
}
