/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Vector3PropMediator.hh"
#include "../SceneObject.hh"
#include "../props/Vector3PropBase.hh"

#include <cassert>
#include <boost/bind.hpp>
#include <limits>


using namespace peekabot;
using namespace peekabot::gui;


Vector3PropMediator::Vector3PropMediator(
    PropInspectorController &pi, SceneObject *obj,
    Vector3PropBase *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0),
      m_lbl(0)
{
    // Create and init widget
    post(
        boost::bind(&Vector3PropMediator::create_widget, this,
                    m_obj->get_prop_name(m_prop_key),
                    any_cast<Eigen::Vector3f>(m_prop->get(m_obj)),
                    m_prop->is_read_only()));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&Vector3PropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void Vector3PropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(
        boost::bind(&Vector3PropMediator::destroy_widget, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&Vector3PropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void Vector3PropMediator::create_widget(
    std::string prop_name,
    Eigen::Vector3f val, bool read_only)
{
    assert( !m_lbl );

    m_hbox = new Gtk::HBox();
    m_hbox->set_spacing(3);

    for( int i = 0; i < 3; ++i )
    {
        m_xyz[i] = new Gtk::SpinButton();
        m_xyz[i]->set_numeric(true);
        m_xyz[i]->set_range(
            -std::numeric_limits<float>::max(),
            std::numeric_limits<float>::max());
        m_xyz[i]->set_value(val(i));
        m_xyz[i]->set_width_chars(3);
        m_xyz[i]->set_increments(0.1, 1);
        m_xyz[i]->set_digits(2);
        m_xyz_set_conn[i] = m_xyz[i]->signal_value_changed().connect(
            sigc::mem_fun(*this, &Vector3PropMediator::on_widget_set));
        if( read_only )
            m_xyz[i]->set_sensitive(false);

        m_hbox->pack_start(*m_xyz[i], true, true);
    }

    m_lbl = new Gtk::Label(prop_name + ":", 1.0, 0.5);

    add_prop_widgets(m_lbl, m_hbox);
}


// Executed in the GUI thread
void Vector3PropMediator::destroy_widget()
{
    assert( m_lbl );

    erase_prop_widgets(m_lbl, m_hbox);
    delete m_hbox;
    m_hbox = 0;
    delete m_lbl;
    m_lbl = 0;

    for( int i = 0; i < 3; ++i )
    {
        delete m_xyz[i];
        m_xyz[i] = 0;
    }

    delete this;
}


// Executed in the server thread
void Vector3PropMediator::on_prop_set()
{
    Eigen::Vector3f val = any_cast<Eigen::Vector3f>(m_prop->get(m_obj));
    post(
        boost::bind(
            &Vector3PropMediator::update_widget, this, val));
}


// Executed in the GUI thread
void Vector3PropMediator::update_widget(Eigen::Vector3f val)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    for( int i = 0; i < 3; ++i )
    {
        m_xyz_set_conn[i].block();
        m_xyz[i]->set_value(val(i));
        m_xyz_set_conn[i].unblock();
    }
}


// Executed in the GUI thread
void Vector3PropMediator::on_widget_set()
{
    ++m_n_updates_queued;

    Eigen::Vector3f val;
    for( int i = 0; i < 3; ++i )
        val(i) = m_xyz[i]->get_value();
    set_prop(m_object_id, m_prop_key, val);
}
