/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_ARRAY_BUFFER_HH_INCLUDED
#define PEEKABOT_RENDERER_ARRAY_BUFFER_HH_INCLUDED


#include <GL/glew.h>
#include <cstddef>
#include <boost/cstdint.hpp>
#include <cassert>


namespace peekabot
{
    class ArrayBuffer
    {
        friend class ReadOnlyMapping;
        friend class WriteOnlyMapping;
        friend class ReadWriteMapping;

    public:
        enum UsageHint
        {
            STATIC_DRAW  = 0,
            DYNAMIC_DRAW = 1,
            STREAM_DRAW  = 2
        };

        explicit ArrayBuffer(bool force_no_vbos, bool index_buffer);

        ArrayBuffer(const ArrayBuffer &other);

        virtual ~ArrayBuffer();

        void set_usage_hint(UsageHint usage);

        UsageHint get_usage_hint() const { return m_usage; }

	void set_data(const void *data, std::size_t n);

	void set_sub_data(const void *data, std::size_t n, std::size_t off);

	void get_data(void *data, std::size_t n, std::size_t off) const;

	inline std::size_t size() const { return m_size; }

        inline bool empty() const { return size() == 0; }

        void resize(std::size_t n);

        void grow_back(std::size_t n);

        void grow_front(std::size_t n);


        void *map(bool write_only);

        const void *map() const;

        void unmap() const;

        inline bool is_mapped() const { return m_is_mapped; }


        template<typename T>
        class ReadOnlyMapping
        {
        public:
            typedef T value_type;

            ReadOnlyMapping(
                const ArrayBuffer &buf,
                std::size_t off = 0, std::size_t padding = 0)
                : m_buf(buf),
                  m_stride(sizeof(T)+padding)
            {
                m_p = reinterpret_cast<const boost::uint8_t *>(m_buf.map()) + off;
            }

            ~ReadOnlyMapping()
            {
                m_buf.unmap();
            }

            inline const value_type &operator[](std::size_t i)
            {
                return *(const T *)(m_p + i*m_stride);
            }

            inline const value_type *get() { return (const value_type *)m_p; }

        private:
            const ArrayBuffer &m_buf;

            const std::size_t m_stride;

            const boost::uint8_t *m_p;
        };


        template<typename T>
        class WriteOnlyMapping
        {
        public:
            typedef T value_type;

            class Proxy
            {
                friend class WriteOnlyMapping;

                explicit inline Proxy(value_type &x) : m_x(x) {}

                Proxy &operator=(const Proxy &);

            public:
                inline const value_type &operator=(const value_type &x)
                {
                    return m_x = x;
                }

            private:
                value_type &m_x;
            };

            WriteOnlyMapping(
                ArrayBuffer &buf,
                std::size_t off = 0, std::size_t padding = 0)
                : m_buf(buf),
                  m_stride(sizeof(T)+padding)
            {
                m_p = reinterpret_cast<boost::uint8_t *>(m_buf.map(true)) + off;
            }

            ~WriteOnlyMapping()
            {
                m_buf.unmap();
            }

            inline Proxy operator[](std::size_t i)
            {
                return Proxy(*(T *)(m_p + i*m_stride));
            }

            inline value_type *get() { return (value_type *)m_p; }

        private:
            ArrayBuffer &m_buf;

            const std::size_t m_stride;

            boost::uint8_t *m_p;
        };


        template<typename T>
        class ReadWriteMapping
        {
        public:
            typedef T value_type;

            ReadWriteMapping(
                ArrayBuffer &buf,
                std::size_t off = 0, std::size_t padding = 0)
                : m_buf(buf),
                  m_stride(sizeof(T)+padding)
            {
                m_p = reinterpret_cast<boost::uint8_t *>(m_buf.map(false)) + off;
            }

            ~ReadWriteMapping()
            {
                m_buf.unmap();
            }

            inline value_type &operator[](std::size_t i)
            {
                return *(T *)(m_p + i*m_stride);
            }

            inline value_type *get() { return (value_type *)m_p; }

        private:
            ArrayBuffer &m_buf;

            const std::size_t m_stride;

            boost::uint8_t *m_p;
        };


    protected:
        void bind() const;

        void unbind() const;

        inline bool is_bound() const
        {
            assert( m_uses_vbos );
            if( m_is_index_buffer )
                return m_vbo_handle == ms_bound_index_buffer;
            else
                return m_vbo_handle == ms_bound_vertex_buffer;
        }

        inline bool uses_vbos() const { return m_uses_vbos; }

        void clear_index_buffer_vbo_binding() const;

        void clear_vertex_buffer_vbo_binding() const;

        const boost::uint8_t *get_data() const { return m_data; }

    private:
        void *map_vbo(bool write_only);

        const void *map_vbo() const;

        void unmap_vbo() const;

        GLenum get_gl_usage_hint() const;

        inline GLenum get_gl_buffer_type() const
        {
            return m_is_index_buffer ?
                GL_ELEMENT_ARRAY_BUFFER_ARB : GL_ARRAY_BUFFER_ARB;
        }

    private:
        typedef GLuint VBOHandle;

        union
        {
            VBOHandle m_vbo_handle;
            boost::uint8_t *m_data;
        };

        std::size_t m_size;

        const bool m_is_index_buffer:1;

        const bool m_uses_vbos:1;

        mutable bool m_is_mapped:1;

        UsageHint m_usage:2;

        static VBOHandle ms_bound_index_buffer;

        static VBOHandle ms_bound_vertex_buffer;
    };
}


#endif // PEEKABOT_RENDERER_ARRAY_BUFFER_HH_INCLUDED
